/*
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Authors: Waiman Long <waiman.long@hp.com>
 *
 * WARNING: Running this test for too long on too many CPUs may crash the
 * 	    kernel.
 *
 * This kernel module enables us to evaluate the contention behavior of locks.
 *
 * The following sysfs variables will be created:
 * 1) /sys/kernel/locktest/c_count
 * 2) /sys/kernel/locktest/i_count
 * 3) /sys/kernel/locktest/l_count
 * 4) /sys/kernel/locktest/p_count
 * 5) /sys/kernel/locktest/locktype
 * 6) /sys/kernel/locktest/rw_ratio
 * 7) /sys/kernel/locktest/load		# load type
 * 8) /sys/kernel/locktest/etime	# elapsed time in ticks
 */
#include <linux/module.h>	// included for all kernel modules
#include <linux/kernel.h>	// included for KERN_INFO
#include <linux/init.h>		// included for __init and __exit macros
#include <linux/kobject.h>
#include <linux/sysfs.h>
#include <linux/string.h>
#include <linux/atomic.h>

static int c_count ;	/* Contention count	*/
static int i_count ;	/* Iteration count	*/
static int l_count ;	/* Load count		*/
static int p_count ;	/* Pause count		*/
static int locktype;	/* Lock type		*/
static int loadtype;	/* Load type		*/
static int rw_ratio;	/* RW (n:1) ratio	*/

/*
 * Show & store macros
 */
#define	SHOW_AND_STORE(v)						\
	static ssize_t v ## _show(struct kobject *kobj,			\
				struct kobj_attribute *attr, char *buf)	\
	{								\
		return sprintf(buf, "%d\n", v);				\
	}								\
	static ssize_t v ## _store(struct kobject *kobj,		\
				struct kobj_attribute *attr,		\
				const char *buf, size_t count)		\
	{								\
		sscanf(buf, "%du", &v);					\
		return count;						\
	}

/*
 * Lock type
 */
#define	LOCK_SPINLOCK	0
#define	LOCK_RWLOCK	1

/*
 * Load type
 * 1) Standalone - lock & protected data in separate cachelines
 * 2) Embedded   - lock & protected data in the same cacheline
 */
#define	LOAD_STANDALONE	0
#define	LOAD_EMBEDDED	1

/*
 * External functions
 */
extern void *__kmalloc_node(size_t size, gfp_t flags, int node);

/*
 * Locks
 */
struct load {
	int c1, c2;
};

struct s_lock {
	struct {
		spinlock_t  lock;
		struct load data;
	} ____cacheline_aligned_in_smp;
};

struct rw_lock {
	struct {
		rwlock_t    lock;
		struct load data;
	}____cacheline_aligned_in_smp;
};

static struct s_lock  *slock;
static struct rw_lock *rwlock;

/*
 * Show & store methods and attributes
 */
SHOW_AND_STORE(c_count) ;
SHOW_AND_STORE(i_count) ;
SHOW_AND_STORE(l_count) ;
SHOW_AND_STORE(p_count) ;
SHOW_AND_STORE(locktype);
SHOW_AND_STORE(rw_ratio);
SHOW_AND_STORE(loadtype);

static struct kobj_attribute c_count_attribute =
	__ATTR(c_count, 0666, c_count_show, c_count_store);
static struct kobj_attribute i_count_attribute =
	__ATTR(i_count, 0666, i_count_show, i_count_store);
static struct kobj_attribute l_count_attribute =
	__ATTR(l_count, 0666, l_count_show, l_count_store);
static struct kobj_attribute p_count_attribute =
	__ATTR(p_count, 0666, p_count_show, p_count_store);
static struct kobj_attribute locktype_attribute =
	__ATTR(locktype, 0666, locktype_show, locktype_store);
static struct kobj_attribute rw_ratio_attribute =
	__ATTR(rw_ratio, 0666, rw_ratio_show, rw_ratio_store);
static struct kobj_attribute loadtype_attribute =
	__ATTR(loadtype, 0666, loadtype_show, loadtype_store);


static ssize_t etime_store(struct kobject *kobj, struct kobj_attribute *attr,
			   const char *buf, size_t count)
{
	/* Dummy method */
	return count;
}

static noinline int load(struct load *data, int ltype, int write, int lcnt)
{
	if (ltype == LOAD_STANDALONE) {
		for (; lcnt > 0; lcnt--)
			cpu_relax();
	} else if (write) {
		for (; lcnt > 0; lcnt--) {
			ACCESS_ONCE(data->c1) += lcnt  ;
			ACCESS_ONCE(data->c2) += 2*lcnt;
		}
	} else {
		int sum;
		for (sum = 0; lcnt > 0; lcnt--)
			sum += ACCESS_ONCE(data->c1) + ACCESS_ONCE(data->c2);
		return sum;
	}
	return 0;
}

static noinline void test_spinlock(void)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int j		;

	for ( ; i > 0 ; i--) {
		spin_lock(&slock->lock);
		load(&slock->data, l, 1, c);
		spin_unlock(&slock->lock);
		for (j = p ; j > 0 ; j--)
			cpu_relax();
	}
}

static noinline void test_rwlock(void)
{
	int i = i_count ;
	int p = p_count ;
	int c = l_count ;
	int l = loadtype;
	int r = rw_ratio;
	int j, k	;

	for (j = 0 ; i > 0 ; i--, j++) {
		if (j >= r) {
			/* Write lock */
			j = -1;
			write_lock(&rwlock->lock);
			load(&slock->data, l, 1, c);
			write_unlock(&rwlock->lock);
		} else {
			/* Read lock */
			read_lock(&rwlock->lock);
			load(&slock->data, l, 0, c);
			read_unlock(&rwlock->lock);
		}
		for (k = p ; k > 0 ; k--)
			cpu_relax();
	}
}

static ssize_t etime_show(struct kobject *kobj, struct kobj_attribute *attr,
			  char *buf)
{
	unsigned long	start, stop ;

	atomic_dec((atomic_t *)&c_count);

	/* Wait until the count reaches 0 */
	while (ACCESS_ONCE(c_count))
		cpu_relax();

	start = jiffies;
	if (locktype == LOCK_SPINLOCK)
		test_spinlock();
	else
		test_rwlock();
	stop = jiffies;
	return sprintf(buf, "%ld\n", stop - start);
}

static struct kobj_attribute etime_attribute =
	__ATTR(etime, 0666, etime_show, etime_store);

static struct attribute *attrs[] = {
	&c_count_attribute.attr ,
	&i_count_attribute.attr ,
	&l_count_attribute.attr ,
	&p_count_attribute.attr ,
	&locktype_attribute.attr  ,
	&loadtype_attribute.attr  ,
	&rw_ratio_attribute.attr,
	&etime_attribute.attr   ,
	NULL
};

static struct attribute_group attr_group = {
        .attrs = attrs,
};

static struct kobject *locktest_kobj;

/*
 * Module init function
 */
static int __init locktest_init(void)
{
	int retval;

	locktest_kobj = kobject_create_and_add("locktest", kernel_kobj);
	if (!locktest_kobj)
		return -ENOMEM;

	retval = sysfs_create_group(locktest_kobj, &attr_group);
	if (retval) {
		kobject_put(locktest_kobj);
		return retval;
	}

	/*
	 * Allocate node 0 memory for locks
	 */
	slock  = __kmalloc_node(sizeof(*slock), GFP_KERNEL, 0);
	rwlock = __kmalloc_node(sizeof(*rwlock), GFP_KERNEL, 0);
	if (!slock || !rwlock) {
		printk(KERN_WARNING "locktest: __kmalloc_node failed!\n");
		return -ENOMEM;
	}
	slock->lock = __SPIN_LOCK_UNLOCKED(slock->lock);
	rwlock_init(&rwlock->lock);

	printk(KERN_INFO "locktest module loaded!\n");

	// Non-zero return means that the module couldn't be loaded.
	return retval;
}

static void __exit locktest_cleanup(void)
{
	printk(KERN_INFO "locktest module unloaded.\n");
	kobject_put(locktest_kobj);
}

module_init(locktest_init);
module_exit(locktest_cleanup);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Waiman Long");
MODULE_DESCRIPTION("Lock testing module");
