/*
    XenBus C stubs for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdint.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>

#include <xenctrl.h>
#include <xen/io/xs_wire.h>

#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/memory.h>
#include <caml/alloc.h>

/* Memory barrier */
value mb_c (value dummy)
{
	CAMLparam1 (dummy);

    asm volatile ( "lock; addl $0,0(%%esp)" : : : "memory" );

	CAMLreturn (Val_unit);
}

/* Map a file */
value mmap_c (value fd_v)
{
	CAMLparam1 (fd_v);

	int fd = Int_val (fd_v);
	long pagesize = getpagesize();
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) mmap(NULL, pagesize, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);

	CAMLreturn (rv);
}

/* Unmap a file */
value munmap_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	long pagesize = getpagesize();

	CAMLreturn (Val_int (munmap(intf, pagesize)));
}

/* Map a foreign page */
value xc_map_foreign_range_c (value xc_handle_v, value domid_v, value mfn_v)
{
	CAMLparam3 (xc_handle_v, domid_v, mfn_v);

	int xc_handle = Int_val (xc_handle_v);
	long pagesize = getpagesize();
	uint32_t domid = (uint32_t)(Int_val (domid_v));
	unsigned long mfn = (unsigned long)(Int_val (mfn_v));
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) xc_map_foreign_range(xc_handle, domid, pagesize, PROT_READ|PROT_WRITE, mfn);

	CAMLreturn (rv);
}

value get_index_c (value index_v)
{
	CAMLparam1 (index_v);

	uint32_t i = *(uint32_t *)(Field (index_v, 0));

	CAMLreturn (caml_copy_int32(i));
}

value set_index_c (value index_v, value val_v)
{
	CAMLparam2 (index_v, val_v);

	uint32_t i = Int32_val (val_v);
	*(uint32_t *)(Field (index_v, 0)) = i;

	CAMLreturn (Val_unit);
}

value init_req_ring_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->req);

	CAMLreturn (rv);
}

value init_rsp_ring_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->rsp);

	CAMLreturn (rv);
}

value init_req_cons_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->req_cons);

	CAMLreturn (rv);
}

value init_req_prod_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->req_prod);

	CAMLreturn (rv);
}

value init_rsp_cons_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv =  alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->rsp_cons);

	CAMLreturn (rv);
}

value init_rsp_prod_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->rsp_prod);

	CAMLreturn (rv);
}

/* Read from a ring buffer */
value read_ring_c (value ring_v, value ring_ofs_v, value buff_v, value buff_ofs_v, value len_v)
{
	CAMLparam5 (ring_v, ring_ofs_v, buff_v, buff_ofs_v, len_v);

	char *ring = (char *)(Field (ring_v, 0));
	char *buff = String_val (buff_v);
	int ring_ofs = Int_val (ring_ofs_v);
	int buff_ofs = Int_val (buff_ofs_v);
	int len = Int_val (len_v);
	int i;

	for (i = 0; i < len; i++) {
		buff[buff_ofs + i] = ring[ring_ofs + i];
	}

	CAMLreturn (Val_unit);
}

/* Write to a ring buffer */
value write_ring_c (value ring_v, value ring_ofs_v, value buff_v, value buff_ofs_v, value len_v)
{
	CAMLparam5 (ring_v, ring_ofs_v, buff_v, buff_ofs_v, len_v);

	char *ring = (char *)(Field (ring_v, 0));
	char *buff = String_val (buff_v);
	int ring_ofs = Int_val (ring_ofs_v);
	int buff_ofs = Int_val (buff_ofs_v);
	int len = Int_val (len_v);
	int i;

	for (i = 0; i < len; i++) {
		ring[ring_ofs + i] = buff[buff_ofs + i];
	}

	CAMLreturn (Val_unit);
}
