(* 
    Options for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Options type *)
type t = {
  fork : bool;
  output_pid : bool;
  domain_init : bool;
  separate_domain : bool;
  pid_file : string;
  trace_file : string;
  recovery : bool;
  verbose : bool;
  quota_num_entries_per_domain : int;
  quota_max_entry_size : int;
  quota_num_watches_per_domain : int;
  quota_max_transaction : int
}

(* Usage message header *)
let usage = "Usage:\n  xenstored <options>\n\nwhere options may include:\n"

(* Parse command-line options *)
let parse () =
  (* Default options *)
  let fork = ref true
  and output_pid = ref false
  and domain_init = ref true
  and pid_file = ref ""
  and trace_file = ref ""
  and recovery = ref true
  and verbose = ref false
  and separate_domain = ref false
  and quota_num_entries_per_domain = ref 1000
  and quota_max_entry_size = ref 2048
  and quota_num_watches_per_domain = ref 128
  and quota_max_transaction = ref 10 in
  
  (* Command-line arguments list *)
  let spec_list = Arg.align [
      ("--no-domain-init", Arg.Clear domain_init, " to state that xenstored should not initialise dom0,");
      ("--pid-file", Arg.Set_string pid_file, "<file> giving a file for the daemon's pid to be written,");
      ("--no-fork", Arg.Clear fork, " to request that the daemon does not fork,");
      ("--output-pid", Arg.Set output_pid, " to request that the pid of the daemon is output,");
      ("--trace-file", Arg.String (fun s -> trace_file := s; Trace.traceout := Some (open_out s)), "<file> giving the file for logging,");
      ("--entry-nb", Arg.Set_int quota_num_entries_per_domain, "<nb> limit the number of entries per domain,");
      ("--entry-size", Arg.Set_int quota_max_entry_size, "<size> limit the size of entry per domain,");
      ("--entry-watch", Arg.Set_int quota_num_watches_per_domain,"<nb> limit the number of watches per domain,");
      ("--transaction", Arg.Set_int quota_max_transaction, "<nb> limit the number of transaction allowed per domain,");
      ("--no-recovery", Arg.Clear recovery, " to request that no recovery should be attempted when the store is corrupted (debug only),");
      ("--preserve-local", Arg.Unit (fun () -> ()), " to request that /local is preserved on start-up,");
      ("--verbose", Arg.Set verbose, " to request verbose execution.");
      ("--separate-dom", Arg.Set separate_domain, " xenstored runs in it's own domain.");
      ] in
  
  (* Parse command-line arguments *)
  Arg.parse spec_list Os.parse_option usage;
  
  (* Set and return chosen options *)
  {
    fork = !fork;
    output_pid = !output_pid;
    domain_init = !domain_init;
    separate_domain = !separate_domain;
    pid_file = !pid_file;
    trace_file = !trace_file;
    recovery = !recovery;
    verbose = !verbose;
    quota_num_entries_per_domain = !quota_num_entries_per_domain;
    quota_max_entry_size = !quota_max_entry_size;
    quota_num_watches_per_domain = !quota_num_watches_per_domain;
    quota_max_transaction = !quota_max_transaction
  }

let check_options options =
  if not options.domain_init && options.separate_domain then Utils.barf_perror "Incompatible options";
  if options.fork then Os.daemonise ();
  if options.pid_file <> Constants.null_string then Os.write_pid_file options.pid_file;
  if options.output_pid then (Printf.printf "%d\n" (Os.get_pid ()); flush stdout)
