xenstore/connection.ml                                                                              0000644 0001750 0001750 00000007617 11146404050 014764  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Connections for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

class buffer length =
object (self)
  val m_buffer = String.make length Constants.null_char
  val mutable m_position = 0
  method private position = m_position
  method buffer = String.copy m_buffer
  method clear =
    String.blit (String.make self#length Constants.null_char) 0 m_buffer 0 self#length;
    m_position <- 0
  method length = String.length m_buffer
  method remaining = self#length - self#position
  method write data =
    let length = (String.length data) in
    String.blit data 0 m_buffer m_position length;
    m_position <- m_position + length
end

class buffered_message =
object (self)
  val m_header = new buffer Message.header_size
  val mutable m_payload = new buffer 0
  method allocate_payload length = m_payload <- new buffer length
  method clear =
    self#header#clear;
    self#allocate_payload 0
  method header = m_header
  method in_header = self#header#remaining <> 0
  method in_payload = self#payload#remaining <> 0
  method message =
    if self#in_header
    then Message.null_message
    else (
      let header = Message.deserialise_header self#header#buffer in
      let payload = if self#payload#length = 0 then String.make header.Message.length Constants.null_char else self#payload#buffer in
      Message.make header.Message.message_type header.Message.transaction_id header.Message.request_id payload
    )
  method payload = m_payload
end

class connection (interface : Interface.interface) =
object (self)
  val m_input_buffer = new buffered_message
  val m_interface = interface
  method private interface = m_interface
  method private input_buffer = m_input_buffer
  method private read_buffer buffer =
    let read_buffer = String.make buffer#remaining Constants.null_char in
    let bytes = self#interface#read read_buffer 0 (String.length read_buffer) in
    if bytes < 0
    then raise (Constants.Xs_error (Constants.EIO, "Connection.connection#read_buffer", "Error reading from interface"))
    else (buffer#write (String.sub read_buffer 0 bytes); buffer#remaining = 0)
  method private write_buffer buffer offset =
    let length = String.length buffer in
    let bytes_written = self#interface#write buffer offset (length - offset) in
    if offset + bytes_written < length then self#write_buffer buffer (offset + bytes_written)
  method can_read = self#interface#can_read
  method can_write = self#interface#can_write
  method destroy = self#interface#destroy
  method read =
    let input = self#input_buffer in
    if input#in_header && self#read_buffer input#header
    then (
      let length = input#message.Message.header.Message.length in
      if length > Constants.payload_max
      then raise (Constants.Xs_error (Constants.EIO, "Connection.connection#read", "Payload too big"))
      else input#allocate_payload length
    );
    if (not input#in_header && not input#in_payload) || (input#in_payload && self#read_buffer input#payload)
    then (
      let message = input#message in
      input#clear;
      Some (message)
    )
    else None
  method write message = self#write_buffer ((Message.serialise_header message.Message.header) ^ message.Message.payload) 0
end
                                                                                                                 xenstore/constants.ml                                                                               0000644 0001750 0001750 00000004145 11146404050 014632  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Constants for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let path_max = 4096
let absolute_path_max = 3072
let relative_path_max = 2048
let payload_max = 4096

(* domain_id_self is used in certain contexts to refer to oneself *)
let domain_id_self = 0x7FF0

(* The prefix character that indicates a watch event *)
let event_char = '@'

let null_char = char_of_int 0
let null_string = String.make 0 null_char
let null_file_descr = - 1

let payload_false = "F"
let payload_true = "T"

let virq_dom_exc = 3

(* Error type *)
type error =
  | EINVAL
  | EACCES
  | EEXIST
  | EISDIR
  | ENOENT
  | ENOMEM
  | ENOSPC
  | EIO
  | ENOTEMPTY
  | ENOSYS
  | EROFS
  | EBUSY
  | EAGAIN
  | EISCONN
  (* XXX: Hack to fix violation of errors specified in protocol *)
  | E2BIG
  | EPERM

(* Return the string representation of an error *)
let error_message error =
  match error with
  | EINVAL -> "EINVAL"
  | EACCES -> "EACCES"
  | EEXIST -> "EEXIST"
  | EISDIR -> "EISDIR"
  | ENOENT -> "ENOENT"
  | ENOMEM -> "ENOMEM"
  | ENOSPC -> "ENOSPC"
  | EIO -> "EIO"
  | ENOTEMPTY -> "ENOTEMPTY"
  | ENOSYS -> "ENOSYS"
  | EROFS -> "EROFS"
  | EBUSY -> "EBUSY"
  | EAGAIN -> "EAGAIN"
  | EISCONN -> "EISCONN"
  (* XXX: Hack to fix violation of errors specified in protocol *)
  | E2BIG -> "E2BIG"
  | EPERM -> "EPERM"

(* Error exception *)
exception Xs_error of error * string * string
                                                                                                                                                                                                                                                                                                                                                                                                                           xenstore/domain.ml                                                                                  0000644 0001750 0001750 00000010760 11146404050 014065  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Domains for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let xc_handle = Eventchan.xc_interface_open ()

class domain (id : int) (connection : Connection.connection) =
object (self)
  val m_id = id
  val m_connection = connection
  val mutable m_input_list = []
  val mutable m_output_list = []
  val mutable m_dying = false
  val mutable m_shutdown = false
  method private connection = m_connection
  method private input_list = m_input_list
  method private output_list = m_output_list
  method add_input_message message = m_input_list <- m_input_list @ [ message ]
  method add_output_message message = m_output_list <- m_output_list @ [ message ]
  method can_read = self#connection#can_read
  method can_write = self#has_output_message && self#connection#can_write
  method destroy = self#connection#destroy
  method dying = m_dying <- true
  method has_input_message = List.length self#input_list > 0
  method has_output_message = List.length self#output_list > 0
  method id = m_id
  method input_message =
    let message = List.hd self#input_list in
    m_input_list <- List.tl m_input_list;
    message
  method input_messages = self#input_list
  method is_dying = m_dying
  method is_shutdown = m_shutdown
  method output_message =
    let message = List.hd self#output_list in
    m_output_list <- List.tl m_output_list;
    message
  method output_messages = self#output_list
  method read = match self#connection#read with Some (message) -> self#add_input_message message | None -> ()
  method shutdown = m_shutdown <- true
  method write = self#connection#write self#output_message
end

class domains =
object (self)
  val m_dominfo = Dominfo.init ()
  val m_entries = Hashtbl.create 8
  val mutable m_domains : domain list = []
  method private check domain =
    if Dominfo.info self#dominfo xc_handle domain#id = 1 && Dominfo.domid self#dominfo = domain#id
    then (
      if (Dominfo.crashed self#dominfo || Dominfo.shutdown self#dominfo) && not domain#is_shutdown then domain#shutdown;
      if Dominfo.dying self#dominfo then domain#dying
    );
    domain#is_dying || domain#is_shutdown
  method private dominfo = m_dominfo
  method private entries = m_entries
  method add domain =
    m_domains <- domain :: m_domains;
    Hashtbl.add self#entries domain#id 0
  method cleanup = List.fold_left (fun domains domain -> if self#check domain then domain :: domains else domains) [] self#domains
  method domains = m_domains
  method entry_count domain_id = Hashtbl.find self#entries domain_id
  method entry_decr domain_id =
    let entries = try pred (Hashtbl.find self#entries domain_id) with Not_found -> 0 in
    Hashtbl.replace self#entries domain_id (if entries < 0 then 0 else entries)
  method entry_incr domain_id = Hashtbl.replace self#entries domain_id (try succ (Hashtbl.find self#entries domain_id) with Not_found -> 1)
  method find_by_id domain_id = List.find (fun domain -> domain#id = domain_id) self#domains
  method remove (domain : domain) =
    m_domains <- List.filter (fun dom -> domain#id <> dom#id) self#domains;
    Hashtbl.remove self#entries domain#id;
    domain#destroy
  method timeout = if List.exists (fun domain -> domain#can_read || domain#can_write) self#domains then 0.0 else - 1.0
end

(* Initialise an unprivileged domain *)
let domu_init id remote_port mfn notify =
  let port = Eventchan.bind_interdomain id remote_port in
  let interface = new Xenbus.xenbus_interface port (Xenbus.map_foreign xc_handle id mfn) in
  let connection = new Connection.connection interface in
  if notify then Eventchan.notify port;
  new domain id connection

(* Check if a domain is unprivileged based on its ID *)
let is_unprivileged_id domain_id =
  domain_id > 0
  
(* Check if a domain is unprivileged *)
let is_unprivileged domain =
  is_unprivileged_id domain#id
                xenstore/dominfo_c.c                                                                                0000644 0001750 0001750 00000005016 11146404050 014363  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /*
    Domain info C stubs for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <unistd.h>

#include <xenctrl.h>
#include <xen/domctl.h>

#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/memory.h>
#include <caml/alloc.h>

/* Initialise a domain's info */
value init_dominfo_c (value dummy_v)
{
	CAMLparam1 (dummy_v);

	value dominfo_v = alloc (Abstract_tag, 1);
	Field (dominfo_v, 0) = (value) malloc (sizeof(xc_dominfo_t));

	CAMLreturn (dominfo_v);
}

/* Return a domain's info */
value xc_domain_getinfo_c (value fd_v, value domid_v, value max_doms_v, value dominfo_v)
{
	CAMLparam4 (fd_v, domid_v, max_doms_v, dominfo_v);

	int fd = Int_val (fd_v);
	uint32_t domid = (uint32_t)(Int_val (domid_v));
	unsigned int max_doms = Int_val (max_doms_v);
	xc_dominfo_t *dominfo = (xc_dominfo_t *)(Field (dominfo_v, 0));

	CAMLreturn (Val_int (xc_domain_getinfo(fd, domid, max_doms, dominfo)));
}

/* Return a domain's crashed state */
value get_crashed_c (value dominfo_v)
{
	CAMLparam1 (dominfo_v);

	xc_dominfo_t *dominfo = (xc_dominfo_t *)(Field (dominfo_v, 0));

	CAMLreturn (caml_copy_int32(dominfo->crashed));
}

/* Return a domain's ID */
value get_domid_c (value dominfo_v)
{
	CAMLparam1 (dominfo_v);

	xc_dominfo_t *dominfo = (xc_dominfo_t *)(Field (dominfo_v, 0));

	CAMLreturn (caml_copy_int32(dominfo->domid));
}

/* Return a domain's dying state */
value get_dying_c (value dominfo_v)
{
	CAMLparam1 (dominfo_v);

	xc_dominfo_t *dominfo = (xc_dominfo_t *)(Field (dominfo_v, 0));

	CAMLreturn (caml_copy_int32(dominfo->dying));
}

/* Return a domain's shutdown state */
value get_shutdown_c (value dominfo_v)
{
	CAMLparam1 (dominfo_v);

	xc_dominfo_t *dominfo = (xc_dominfo_t *)(Field (dominfo_v, 0));

	CAMLreturn (caml_copy_int32(dominfo->shutdown));
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  xenstore/dominfo.ml                                                                                 0000644 0001750 0001750 00000003122 11146404050 014243  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Domain info for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type t;;

external get_crashed32 : t -> int32 = "get_crashed_c";;
external get_domid32 : t -> int32 = "get_domid_c";;
external get_dying32 : t -> int32 = "get_dying_c";;
external get_shutdown32 : t -> int32 = "get_shutdown_c";;
external init : unit -> t = "init_dominfo_c";;
external xc_domain_getinfo : int -> int -> int -> t -> int = "xc_domain_getinfo_c";;

(* Return crashed state *)
let crashed dominfo =
  get_crashed32 dominfo <> 0l

(* Return domain ID *)
let domid dominfo =
  Int32.to_int (get_domid32 dominfo)

(* Return dying state *)
let dying dominfo =
  get_dying32 dominfo <> 0l

(* Return domain info *)
let info dominfo xc_handle id =
  xc_domain_getinfo xc_handle id 1 dominfo

(* Return shutdown state *)
let shutdown dominfo =
  get_shutdown32 dominfo <> 0l
                                                                                                                                                                                                                                                                                                                                                                                                                                              xenstore/eventchan_c.c                                                                              0000644 0001750 0001750 00000006745 11146404050 014715  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /*
    Event channel C stubs for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <unistd.h>
#include <xenctrl.h>

#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/memory.h>


/* Bind an interdomain event channel */
value xc_evtchn_bind_interdomain_c (value xce_handle_v, value domid_v, value remote_port_v)
{
    CAMLparam3 (xce_handle_v, domid_v, remote_port_v);

    int xce_handle = Int_val (xce_handle_v);
    int domid = Int_val (domid_v);
    uint32_t remote_port = (uint32_t)(Int_val (remote_port_v));

    CAMLreturn (Val_int (xc_evtchn_bind_interdomain (xce_handle, domid, remote_port)));
}

/* Bind the VIRQ event channel */
value xc_evtchn_bind_virq_c (value xce_handle_v, value virq_v)
{
    CAMLparam2 (xce_handle_v, virq_v);

    int xce_handle = Int_val (xce_handle_v);
    unsigned int virq = Int_val (virq_v);

    CAMLreturn (Val_int (xc_evtchn_bind_virq (xce_handle, virq)));
}

/* Return the event channel file descriptor */
value xc_evtchn_fd_c (value xce_handle_v)
{
    CAMLparam1 (xce_handle_v);

    int xce_handle = Int_val (xce_handle_v);

    CAMLreturn (Val_int (xc_evtchn_fd (xce_handle)));
}

/* Notify an event channel of an event */
value xc_evtchn_notify_c (value xce_handle_v, value port_v)
{
    CAMLparam2 (xce_handle_v, port_v);

    int xce_handle = Int_val (xce_handle_v);
    uint32_t port = (uint32_t)(Int_val (port_v));

    CAMLreturn (Val_int (xc_evtchn_notify (xce_handle, port)));
}

/* Open the event channel */
value xc_evtchn_open_c (value dummy_v)
{
    CAMLparam1 (dummy_v);
    CAMLreturn (Val_int (xc_evtchn_open ()));
}

/* Check an event channel for pending events */
value xc_evtchn_pending_c (value xce_handle_v)
{
    CAMLparam1 (xce_handle_v);

    int xce_handle = Int_val (xce_handle_v);

    CAMLreturn (Val_int (xc_evtchn_pending (xce_handle)));
}

/* Unbind an event channel */
value xc_evtchn_unbind_c (value xce_handle_v, value port_v)
{
    CAMLparam2 (xce_handle_v, port_v);

    int xce_handle = Int_val (xce_handle_v);
    uint32_t port = (uint32_t)(Int_val (port_v));

    CAMLreturn (Val_int (xc_evtchn_unbind (xce_handle, port)));
}

/* Unmask an event channel */
value xc_evtchn_unmask_c (value xce_handle_v, value port_v)
{
    CAMLparam2 (xce_handle_v, port_v);

    int xce_handle = Int_val (xce_handle_v);
    uint32_t port = (uint32_t)(Int_val (port_v));

    CAMLreturn (Val_int (xc_evtchn_unmask (xce_handle, port)));
}

/* Close the XenBus interface */
value xc_interface_close_c (value xc_handle_v)
{
    CAMLparam1 (xc_handle_v);
    CAMLreturn (Val_int (xc_interface_close (Int_val (xc_handle_v))));
}

/* Open the XenBus interface */
value xc_interface_open_c (value dummy_v)
{
    CAMLparam1 (dummy_v);
    CAMLreturn (Val_int (xc_interface_open ()));
}
                           xenstore/eventchan.ml                                                                               0000644 0001750 0001750 00000004675 11146404050 014601  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Event channel for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

external fake_call : unit -> int = "xc_interface_open"
external xc_event_chan_bind_interdomain : int -> int -> int -> int = "xc_evtchn_bind_interdomain_c"
external xc_event_chan_bind_virq : int -> int -> int = "xc_evtchn_bind_virq_c"
external xc_event_chan_fd : int -> int = "xc_evtchn_fd_c"
external xc_event_chan_open : unit -> int = "xc_evtchn_open_c"
external xc_event_chan_notify : int -> int -> int = "xc_evtchn_notify_c"
external xc_event_chan_pending : int -> int = "xc_evtchn_pending_c"
external xc_event_chan_unbind : int -> int -> int = "xc_evtchn_unbind_c"
external xc_event_chan_unmask : int -> int -> int = "xc_evtchn_unmask_c"
external xc_interface_open : unit -> int = "xc_interface_open_c"
external xc_interface_close : int -> int = "xc_interface_close_c"

(* XXX: Force libxenctrl to be compiled in. There must be a better way *)
let fake () =
  fake_call ()

let xce_handle = ref (- 1)

(* Bind a domain to the remove end *)
let bind_interdomain id remote_port =
  xc_event_chan_bind_interdomain !xce_handle id remote_port

(* Bind the virq *)
let bind_virq virq =
  xc_event_chan_bind_virq !xce_handle virq

(* Return the event channel fd *)
let get_channel () =
  xc_event_chan_fd !xce_handle

(* Intialise the event channel *)
let init () =
  xce_handle := xc_event_chan_open ()

(* Notify XenBus *)
let notify port =
  ignore (xc_event_chan_notify !xce_handle port)

(* Check for pending event *)
let pending () =
  xc_event_chan_pending !xce_handle

(* Unbind a XenBus port *)
let unbind port =
  xc_event_chan_unbind !xce_handle port <> - 1

(* Unmask a XenBus port *)
let unmask port =
  xc_event_chan_unmask !xce_handle port
                                                                   xenstore/gpl-2.0.txt                                                                                0000644 0001750 0001750 00000043103 11146404050 014101  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 		    GNU GENERAL PUBLIC LICENSE
		       Version 2, June 1991

 Copyright (C) 1989, 1991 Free Software Foundation, Inc.,
 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 Everyone is permitted to copy and distribute verbatim copies
 of this license document, but changing it is not allowed.

			    Preamble

  The licenses for most software are designed to take away your
freedom to share and change it.  By contrast, the GNU General Public
License is intended to guarantee your freedom to share and change free
software--to make sure the software is free for all its users.  This
General Public License applies to most of the Free Software
Foundation's software and to any other program whose authors commit to
using it.  (Some other Free Software Foundation software is covered by
the GNU Lesser General Public License instead.)  You can apply it to
your programs, too.

  When we speak of free software, we are referring to freedom, not
price.  Our General Public Licenses are designed to make sure that you
have the freedom to distribute copies of free software (and charge for
this service if you wish), that you receive source code or can get it
if you want it, that you can change the software or use pieces of it
in new free programs; and that you know you can do these things.

  To protect your rights, we need to make restrictions that forbid
anyone to deny you these rights or to ask you to surrender the rights.
These restrictions translate to certain responsibilities for you if you
distribute copies of the software, or if you modify it.

  For example, if you distribute copies of such a program, whether
gratis or for a fee, you must give the recipients all the rights that
you have.  You must make sure that they, too, receive or can get the
source code.  And you must show them these terms so they know their
rights.

  We protect your rights with two steps: (1) copyright the software, and
(2) offer you this license which gives you legal permission to copy,
distribute and/or modify the software.

  Also, for each author's protection and ours, we want to make certain
that everyone understands that there is no warranty for this free
software.  If the software is modified by someone else and passed on, we
want its recipients to know that what they have is not the original, so
that any problems introduced by others will not reflect on the original
authors' reputations.

  Finally, any free program is threatened constantly by software
patents.  We wish to avoid the danger that redistributors of a free
program will individually obtain patent licenses, in effect making the
program proprietary.  To prevent this, we have made it clear that any
patent must be licensed for everyone's free use or not licensed at all.

  The precise terms and conditions for copying, distribution and
modification follow.

		    GNU GENERAL PUBLIC LICENSE
   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION

  0. This License applies to any program or other work which contains
a notice placed by the copyright holder saying it may be distributed
under the terms of this General Public License.  The "Program", below,
refers to any such program or work, and a "work based on the Program"
means either the Program or any derivative work under copyright law:
that is to say, a work containing the Program or a portion of it,
either verbatim or with modifications and/or translated into another
language.  (Hereinafter, translation is included without limitation in
the term "modification".)  Each licensee is addressed as "you".

Activities other than copying, distribution and modification are not
covered by this License; they are outside its scope.  The act of
running the Program is not restricted, and the output from the Program
is covered only if its contents constitute a work based on the
Program (independent of having been made by running the Program).
Whether that is true depends on what the Program does.

  1. You may copy and distribute verbatim copies of the Program's
source code as you receive it, in any medium, provided that you
conspicuously and appropriately publish on each copy an appropriate
copyright notice and disclaimer of warranty; keep intact all the
notices that refer to this License and to the absence of any warranty;
and give any other recipients of the Program a copy of this License
along with the Program.

You may charge a fee for the physical act of transferring a copy, and
you may at your option offer warranty protection in exchange for a fee.

  2. You may modify your copy or copies of the Program or any portion
of it, thus forming a work based on the Program, and copy and
distribute such modifications or work under the terms of Section 1
above, provided that you also meet all of these conditions:

    a) You must cause the modified files to carry prominent notices
    stating that you changed the files and the date of any change.

    b) You must cause any work that you distribute or publish, that in
    whole or in part contains or is derived from the Program or any
    part thereof, to be licensed as a whole at no charge to all third
    parties under the terms of this License.

    c) If the modified program normally reads commands interactively
    when run, you must cause it, when started running for such
    interactive use in the most ordinary way, to print or display an
    announcement including an appropriate copyright notice and a
    notice that there is no warranty (or else, saying that you provide
    a warranty) and that users may redistribute the program under
    these conditions, and telling the user how to view a copy of this
    License.  (Exception: if the Program itself is interactive but
    does not normally print such an announcement, your work based on
    the Program is not required to print an announcement.)

These requirements apply to the modified work as a whole.  If
identifiable sections of that work are not derived from the Program,
and can be reasonably considered independent and separate works in
themselves, then this License, and its terms, do not apply to those
sections when you distribute them as separate works.  But when you
distribute the same sections as part of a whole which is a work based
on the Program, the distribution of the whole must be on the terms of
this License, whose permissions for other licensees extend to the
entire whole, and thus to each and every part regardless of who wrote it.

Thus, it is not the intent of this section to claim rights or contest
your rights to work written entirely by you; rather, the intent is to
exercise the right to control the distribution of derivative or
collective works based on the Program.

In addition, mere aggregation of another work not based on the Program
with the Program (or with a work based on the Program) on a volume of
a storage or distribution medium does not bring the other work under
the scope of this License.

  3. You may copy and distribute the Program (or a work based on it,
under Section 2) in object code or executable form under the terms of
Sections 1 and 2 above provided that you also do one of the following:

    a) Accompany it with the complete corresponding machine-readable
    source code, which must be distributed under the terms of Sections
    1 and 2 above on a medium customarily used for software interchange; or,

    b) Accompany it with a written offer, valid for at least three
    years, to give any third party, for a charge no more than your
    cost of physically performing source distribution, a complete
    machine-readable copy of the corresponding source code, to be
    distributed under the terms of Sections 1 and 2 above on a medium
    customarily used for software interchange; or,

    c) Accompany it with the information you received as to the offer
    to distribute corresponding source code.  (This alternative is
    allowed only for noncommercial distribution and only if you
    received the program in object code or executable form with such
    an offer, in accord with Subsection b above.)

The source code for a work means the preferred form of the work for
making modifications to it.  For an executable work, complete source
code means all the source code for all modules it contains, plus any
associated interface definition files, plus the scripts used to
control compilation and installation of the executable.  However, as a
special exception, the source code distributed need not include
anything that is normally distributed (in either source or binary
form) with the major components (compiler, kernel, and so on) of the
operating system on which the executable runs, unless that component
itself accompanies the executable.

If distribution of executable or object code is made by offering
access to copy from a designated place, then offering equivalent
access to copy the source code from the same place counts as
distribution of the source code, even though third parties are not
compelled to copy the source along with the object code.

  4. You may not copy, modify, sublicense, or distribute the Program
except as expressly provided under this License.  Any attempt
otherwise to copy, modify, sublicense or distribute the Program is
void, and will automatically terminate your rights under this License.
However, parties who have received copies, or rights, from you under
this License will not have their licenses terminated so long as such
parties remain in full compliance.

  5. You are not required to accept this License, since you have not
signed it.  However, nothing else grants you permission to modify or
distribute the Program or its derivative works.  These actions are
prohibited by law if you do not accept this License.  Therefore, by
modifying or distributing the Program (or any work based on the
Program), you indicate your acceptance of this License to do so, and
all its terms and conditions for copying, distributing or modifying
the Program or works based on it.

  6. Each time you redistribute the Program (or any work based on the
Program), the recipient automatically receives a license from the
original licensor to copy, distribute or modify the Program subject to
these terms and conditions.  You may not impose any further
restrictions on the recipients' exercise of the rights granted herein.
You are not responsible for enforcing compliance by third parties to
this License.

  7. If, as a consequence of a court judgment or allegation of patent
infringement or for any other reason (not limited to patent issues),
conditions are imposed on you (whether by court order, agreement or
otherwise) that contradict the conditions of this License, they do not
excuse you from the conditions of this License.  If you cannot
distribute so as to satisfy simultaneously your obligations under this
License and any other pertinent obligations, then as a consequence you
may not distribute the Program at all.  For example, if a patent
license would not permit royalty-free redistribution of the Program by
all those who receive copies directly or indirectly through you, then
the only way you could satisfy both it and this License would be to
refrain entirely from distribution of the Program.

If any portion of this section is held invalid or unenforceable under
any particular circumstance, the balance of the section is intended to
apply and the section as a whole is intended to apply in other
circumstances.

It is not the purpose of this section to induce you to infringe any
patents or other property right claims or to contest validity of any
such claims; this section has the sole purpose of protecting the
integrity of the free software distribution system, which is
implemented by public license practices.  Many people have made
generous contributions to the wide range of software distributed
through that system in reliance on consistent application of that
system; it is up to the author/donor to decide if he or she is willing
to distribute software through any other system and a licensee cannot
impose that choice.

This section is intended to make thoroughly clear what is believed to
be a consequence of the rest of this License.

  8. If the distribution and/or use of the Program is restricted in
certain countries either by patents or by copyrighted interfaces, the
original copyright holder who places the Program under this License
may add an explicit geographical distribution limitation excluding
those countries, so that distribution is permitted only in or among
countries not thus excluded.  In such case, this License incorporates
the limitation as if written in the body of this License.

  9. The Free Software Foundation may publish revised and/or new versions
of the General Public License from time to time.  Such new versions will
be similar in spirit to the present version, but may differ in detail to
address new problems or concerns.

Each version is given a distinguishing version number.  If the Program
specifies a version number of this License which applies to it and "any
later version", you have the option of following the terms and conditions
either of that version or of any later version published by the Free
Software Foundation.  If the Program does not specify a version number of
this License, you may choose any version ever published by the Free Software
Foundation.

  10. If you wish to incorporate parts of the Program into other free
programs whose distribution conditions are different, write to the author
to ask for permission.  For software which is copyrighted by the Free
Software Foundation, write to the Free Software Foundation; we sometimes
make exceptions for this.  Our decision will be guided by the two goals
of preserving the free status of all derivatives of our free software and
of promoting the sharing and reuse of software generally.

			    NO WARRANTY

  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY
FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN
OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES
PROVIDE THE PROGRAM "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED
OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS
TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE
PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,
REPAIR OR CORRECTION.

  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING
WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR
REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,
INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING
OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED
TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY
YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER
PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE
POSSIBILITY OF SUCH DAMAGES.

		     END OF TERMS AND CONDITIONS

	    How to Apply These Terms to Your New Programs

  If you develop a new program, and you want it to be of the greatest
possible use to the public, the best way to achieve this is to make it
free software which everyone can redistribute and change under these terms.

  To do so, attach the following notices to the program.  It is safest
to attach them to the start of each source file to most effectively
convey the exclusion of warranty; and each file should have at least
the "copyright" line and a pointer to where the full notice is found.

    <one line to give the program's name and a brief idea of what it does.>
    Copyright (C) <year>  <name of author>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License along
    with this program; if not, write to the Free Software Foundation, Inc.,
    51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

Also add information on how to contact you by electronic and paper mail.

If the program is interactive, make it output a short notice like this
when it starts in an interactive mode:

    Gnomovision version 69, Copyright (C) year name of author
    Gnomovision comes with ABSOLUTELY NO WARRANTY; for details type `show w'.
    This is free software, and you are welcome to redistribute it
    under certain conditions; type `show c' for details.

The hypothetical commands `show w' and `show c' should show the appropriate
parts of the General Public License.  Of course, the commands you use may
be called something other than `show w' and `show c'; they could even be
mouse-clicks or menu items--whatever suits your program.

You should also get your employer (if you work as a programmer) or your
school, if any, to sign a "copyright disclaimer" for the program, if
necessary.  Here is a sample; alter the names:

  Yoyodyne, Inc., hereby disclaims all copyright interest in the program
  `Gnomovision' (which makes passes at compilers) written by James Hacker.

  <signature of Ty Coon>, 1 April 1989
  Ty Coon, President of Vice

This General Public License does not permit incorporating your program into
proprietary programs.  If your program is a subroutine library, you may
consider it more useful to permit linking proprietary applications with the
library.  If this is what you want to do, use the GNU Lesser General
Public License instead of this License.
                                                                                                                                                                                                                                                                                                                                                                                                                                                             xenstore/interface.ml                                                                               0000644 0001750 0001750 00000002066 11146404050 014556  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Interface for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

class virtual interface =
object
  method virtual can_read : bool
  method virtual can_write : bool
  method virtual destroy : unit
  method virtual read : string -> int -> int -> int
  method virtual write : string -> int -> int -> int
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                          xenstore/list.h                                                                                     0000644 0001750 0001750 00000034253 11146403776 013431  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 #ifndef _LINUX_LIST_H
#define _LINUX_LIST_H
/* Taken from Linux kernel code, but de-kernelized for userspace. */
#include <stddef.h>

#undef LIST_HEAD_INIT
#undef LIST_HEAD
#undef INIT_LIST_HEAD

/*
 * These are non-NULL pointers that will result in page faults
 * under normal circumstances, used to verify that nobody uses
 * non-initialized list entries.
 */
#define LIST_POISON1  ((void *) 0x00100100)
#define LIST_POISON2  ((void *) 0x00200200)

#define container_of(ptr, type, member) ({			\
        typeof( ((type *)0)->member ) *__mptr = (ptr);	\
        (type *)( (char *)__mptr - offsetof(type,member) );})

/*
 * Simple doubly linked list implementation.
 *
 * Some of the internal functions ("__xxx") are useful when
 * manipulating whole lists rather than single entries, as
 * sometimes we already know the next/prev entries and we can
 * generate better code by using them directly rather than
 * using the generic single-entry routines.
 */

struct list_head {
	struct list_head *next, *prev;
};

#define LIST_HEAD_INIT(name) { &(name), &(name) }

#define LIST_HEAD(name) \
	struct list_head name = LIST_HEAD_INIT(name)

#define INIT_LIST_HEAD(ptr) do { \
	(ptr)->next = (ptr); (ptr)->prev = (ptr); \
} while (0)

#define list_top(head, type, member)					  \
({ 									  \
	struct list_head *_head = (head);				  \
	list_empty(_head) ? NULL : list_entry(_head->next, type, member); \
})

/*
 * Insert a new entry between two known consecutive entries. 
 *
 * This is only for internal list manipulation where we know
 * the prev/next entries already!
 */
static inline void __list_add(struct list_head *new,
			      struct list_head *prev,
			      struct list_head *next)
{
	next->prev = new;
	new->next = next;
	new->prev = prev;
	prev->next = new;
}

/**
 * list_add - add a new entry
 * @new: new entry to be added
 * @head: list head to add it after
 *
 * Insert a new entry after the specified head.
 * This is good for implementing stacks.
 */
static inline void list_add(struct list_head *new, struct list_head *head)
{
	__list_add(new, head, head->next);
}

/**
 * list_add_tail - add a new entry
 * @new: new entry to be added
 * @head: list head to add it before
 *
 * Insert a new entry before the specified head.
 * This is useful for implementing queues.
 */
static inline void list_add_tail(struct list_head *new, struct list_head *head)
{
	__list_add(new, head->prev, head);
}

/*
 * Insert a new entry between two known consecutive entries. 
 *
 * This is only for internal list manipulation where we know
 * the prev/next entries already!
 */
static __inline__ void __list_add_rcu(struct list_head * new,
	struct list_head * prev,
	struct list_head * next)
{
	new->next = next;
	new->prev = prev;
	next->prev = new;
	prev->next = new;
}

/**
 * list_add_rcu - add a new entry to rcu-protected list
 * @new: new entry to be added
 * @head: list head to add it after
 *
 * Insert a new entry after the specified head.
 * This is good for implementing stacks.
 */
static __inline__ void list_add_rcu(struct list_head *new, struct list_head *head)
{
	__list_add_rcu(new, head, head->next);
}

/**
 * list_add_tail_rcu - add a new entry to rcu-protected list
 * @new: new entry to be added
 * @head: list head to add it before
 *
 * Insert a new entry before the specified head.
 * This is useful for implementing queues.
 */
static __inline__ void list_add_tail_rcu(struct list_head *new, struct list_head *head)
{
	__list_add_rcu(new, head->prev, head);
}

/*
 * Delete a list entry by making the prev/next entries
 * point to each other.
 *
 * This is only for internal list manipulation where we know
 * the prev/next entries already!
 */
static inline void __list_del(struct list_head * prev, struct list_head * next)
{
	next->prev = prev;
	prev->next = next;
}

/**
 * list_del - deletes entry from list.
 * @entry: the element to delete from the list.
 * Note: list_empty on entry does not return true after this, the entry is
 * in an undefined state.
 */
static inline void list_del(struct list_head *entry)
{
	__list_del(entry->prev, entry->next);
	entry->next = LIST_POISON1;
	entry->prev = LIST_POISON2;
}

/**
 * list_del_rcu - deletes entry from list without re-initialization
 * @entry: the element to delete from the list.
 *
 * Note: list_empty on entry does not return true after this, 
 * the entry is in an undefined state. It is useful for RCU based
 * lockfree traversal.
 *
 * In particular, it means that we can not poison the forward 
 * pointers that may still be used for walking the list.
 */
static inline void list_del_rcu(struct list_head *entry)
{
	__list_del(entry->prev, entry->next);
	entry->prev = LIST_POISON2;
}

/**
 * list_del_init - deletes entry from list and reinitialize it.
 * @entry: the element to delete from the list.
 */
static inline void list_del_init(struct list_head *entry)
{
	__list_del(entry->prev, entry->next);
	INIT_LIST_HEAD(entry); 
}

/**
 * list_move - delete from one list and add as another's head
 * @list: the entry to move
 * @head: the head that will precede our entry
 */
static inline void list_move(struct list_head *list, struct list_head *head)
{
        __list_del(list->prev, list->next);
        list_add(list, head);
}

/**
 * list_move_tail - delete from one list and add as another's tail
 * @list: the entry to move
 * @head: the head that will follow our entry
 */
static inline void list_move_tail(struct list_head *list,
				  struct list_head *head)
{
        __list_del(list->prev, list->next);
        list_add_tail(list, head);
}

/**
 * list_empty - tests whether a list is empty
 * @head: the list to test.
 */
static inline int list_empty(struct list_head *head)
{
	return head->next == head;
}

static inline void __list_splice(struct list_head *list,
				 struct list_head *head)
{
	struct list_head *first = list->next;
	struct list_head *last = list->prev;
	struct list_head *at = head->next;

	first->prev = head;
	head->next = first;

	last->next = at;
	at->prev = last;
}

/**
 * list_splice - join two lists
 * @list: the new list to add.
 * @head: the place to add it in the first list.
 */
static inline void list_splice(struct list_head *list, struct list_head *head)
{
	if (!list_empty(list))
		__list_splice(list, head);
}

/**
 * list_splice_init - join two lists and reinitialise the emptied list.
 * @list: the new list to add.
 * @head: the place to add it in the first list.
 *
 * The list at @list is reinitialised
 */
static inline void list_splice_init(struct list_head *list,
				    struct list_head *head)
{
	if (!list_empty(list)) {
		__list_splice(list, head);
		INIT_LIST_HEAD(list);
	}
}

/**
 * list_entry - get the struct for this entry
 * @ptr:	the &struct list_head pointer.
 * @type:	the type of the struct this is embedded in.
 * @member:	the name of the list_struct within the struct.
 */
#define list_entry(ptr, type, member) \
	container_of(ptr, type, member)

/**
 * list_for_each	-	iterate over a list
 * @pos:	the &struct list_head to use as a loop counter.
 * @head:	the head for your list.
 */
#define list_for_each(pos, head) \
	for (pos = (head)->next; pos != (head); pos = pos->next)

/**
 * list_for_each_prev	-	iterate over a list backwards
 * @pos:	the &struct list_head to use as a loop counter.
 * @head:	the head for your list.
 */
#define list_for_each_prev(pos, head) \
	for (pos = (head)->prev; pos != (head); pos = pos->prev)
        	
/**
 * list_for_each_safe	-	iterate over a list safe against removal of list entry
 * @pos:	the &struct list_head to use as a loop counter.
 * @n:		another &struct list_head to use as temporary storage
 * @head:	the head for your list.
 */
#define list_for_each_safe(pos, n, head) \
	for (pos = (head)->next, n = pos->next; pos != (head); \
		pos = n, n = pos->next)

/**
 * list_for_each_entry	-	iterate over list of given type
 * @pos:	the type * to use as a loop counter.
 * @head:	the head for your list.
 * @member:	the name of the list_struct within the struct.
 */
#define list_for_each_entry(pos, head, member)				\
	for (pos = list_entry((head)->next, typeof(*pos), member);	\
	     &pos->member != (head); 					\
	     pos = list_entry(pos->member.next, typeof(*pos), member))

/**
 * list_for_each_entry_reverse - iterate backwards over list of given type.
 * @pos:	the type * to use as a loop counter.
 * @head:	the head for your list.
 * @member:	the name of the list_struct within the struct.
 */
#define list_for_each_entry_reverse(pos, head, member)			\
	for (pos = list_entry((head)->prev, typeof(*pos), member);	\
	     &pos->member != (head); 					\
	     pos = list_entry(pos->member.prev, typeof(*pos), member))


/**
 * list_for_each_entry_continue -	iterate over list of given type
 *			continuing after existing point
 * @pos:	the type * to use as a loop counter.
 * @head:	the head for your list.
 * @member:	the name of the list_struct within the struct.
 */
#define list_for_each_entry_continue(pos, head, member) 		\
	for (pos = list_entry(pos->member.next, typeof(*pos), member);	\
	     &pos->member != (head);	\
	     pos = list_entry(pos->member.next, typeof(*pos), member))

/**
 * list_for_each_entry_safe - iterate over list of given type safe against removal of list entry
 * @pos:	the type * to use as a loop counter.
 * @n:		another type * to use as temporary storage
 * @head:	the head for your list.
 * @member:	the name of the list_struct within the struct.
 */
#define list_for_each_entry_safe(pos, n, head, member)			\
	for (pos = list_entry((head)->next, typeof(*pos), member),	\
		n = list_entry(pos->member.next, typeof(*pos), member);	\
	     &pos->member != (head); 					\
	     pos = n, n = list_entry(n->member.next, typeof(*n), member))


/* 
 * Double linked lists with a single pointer list head. 
 * Mostly useful for hash tables where the two pointer list head is 
 * too wasteful.
 * You lose the ability to access the tail in O(1).
 */ 

struct hlist_head { 
	struct hlist_node *first; 
}; 

struct hlist_node { 
	struct hlist_node *next, **pprev; 
}; 

#define HLIST_HEAD_INIT { .first = NULL } 
#define HLIST_HEAD(name) struct hlist_head name = {  .first = NULL }
#define INIT_HLIST_HEAD(ptr) ((ptr)->first = NULL) 
#define INIT_HLIST_NODE(ptr) ((ptr)->next = NULL, (ptr)->pprev = NULL)

static __inline__ int hlist_unhashed(struct hlist_node *h) 
{ 
	return !h->pprev;
} 

static __inline__ int hlist_empty(struct hlist_head *h) 
{ 
	return !h->first;
} 

static __inline__ void __hlist_del(struct hlist_node *n) 
{
	struct hlist_node *next = n->next;
	struct hlist_node **pprev = n->pprev;
	*pprev = next;  
	if (next) 
		next->pprev = pprev;
}  

static __inline__ void hlist_del(struct hlist_node *n)
{
	__hlist_del(n);
	n->next = LIST_POISON1;
	n->pprev = LIST_POISON2;
}

/**
 * hlist_del_rcu - deletes entry from hash list without re-initialization
 * @entry: the element to delete from the hash list.
 *
 * Note: list_unhashed() on entry does not return true after this, 
 * the entry is in an undefined state. It is useful for RCU based
 * lockfree traversal.
 *
 * In particular, it means that we can not poison the forward
 * pointers that may still be used for walking the hash list.
 */
static inline void hlist_del_rcu(struct hlist_node *n)
{
	__hlist_del(n);
	n->pprev = LIST_POISON2;
}

static __inline__ void hlist_del_init(struct hlist_node *n) 
{
	if (n->pprev)  {
		__hlist_del(n);
		INIT_HLIST_NODE(n);
	}
}  

#define hlist_del_rcu_init hlist_del_init

static __inline__ void hlist_add_head(struct hlist_node *n, struct hlist_head *h) 
{ 
	struct hlist_node *first = h->first;
	n->next = first; 
	if (first) 
		first->pprev = &n->next;
	h->first = n; 
	n->pprev = &h->first; 
} 

static __inline__ void hlist_add_head_rcu(struct hlist_node *n, struct hlist_head *h) 
{ 
	struct hlist_node *first = h->first;
	n->next = first;
	n->pprev = &h->first; 
	if (first) 
		first->pprev = &n->next;
	h->first = n; 
} 

/* next must be != NULL */
static __inline__ void hlist_add_before(struct hlist_node *n, struct hlist_node *next)
{
	n->pprev = next->pprev;
	n->next = next; 
	next->pprev = &n->next; 
	*(n->pprev) = n;
}

static __inline__ void hlist_add_after(struct hlist_node *n,
				       struct hlist_node *next)
{
	next->next	= n->next;
	*(next->pprev)	= n;
	n->next		= next;
}

#define hlist_entry(ptr, type, member) container_of(ptr,type,member)

/* Cannot easily do prefetch unfortunately */
#define hlist_for_each(pos, head) \
	for (pos = (head)->first; pos; pos = pos->next) 

#define hlist_for_each_safe(pos, n, head) \
	for (pos = (head)->first; n = pos ? pos->next : 0, pos; \
	     pos = n)

/**
 * hlist_for_each_entry	- iterate over list of given type
 * @tpos:	the type * to use as a loop counter.
 * @pos:	the &struct hlist_node to use as a loop counter.
 * @head:	the head for your list.
 * @member:	the name of the hlist_node within the struct.
 */
#define hlist_for_each_entry(tpos, pos, head, member)			 \
	for (pos = (head)->first;					 \
	     pos && ({ tpos = hlist_entry(pos, typeof(*tpos), member); 1;}); \
	     pos = pos->next)

/**
 * hlist_for_each_entry_continue - iterate over a hlist continuing after existing point
 * @tpos:	the type * to use as a loop counter.
 * @pos:	the &struct hlist_node to use as a loop counter.
 * @member:	the name of the hlist_node within the struct.
 */
#define hlist_for_each_entry_continue(tpos, pos, member)		 \
	for (pos = (pos)->next;						 \
	     pos && ({ tpos = hlist_entry(pos, typeof(*tpos), member); 1;}); \
	     pos = pos->next)

/**
 * hlist_for_each_entry_from - iterate over a hlist continuing from existing point
 * @tpos:	the type * to use as a loop counter.
 * @pos:	the &struct hlist_node to use as a loop counter.
 * @member:	the name of the hlist_node within the struct.
 */
#define hlist_for_each_entry_from(tpos, pos, member)			 \
	for (; pos && ({ tpos = hlist_entry(pos, typeof(*tpos), member); 1;}); \
	     pos = pos->next)

/**
 * hlist_for_each_entry_safe - iterate over list of given type safe against removal of list entry
 * @tpos:	the type * to use as a loop counter.
 * @pos:	the &struct hlist_node to use as a loop counter.
 * @n:		another &struct hlist_node to use as temporary storage
 * @head:	the head for your list.
 * @member:	the name of the hlist_node within the struct.
 */
#define hlist_for_each_entry_safe(tpos, pos, n, head, member) 		 \
	for (pos = (head)->first;					 \
	     pos && ({ n = pos->next; 1; }) && 				 \
		({ tpos = hlist_entry(pos, typeof(*tpos), member); 1;}); \
	     pos = n)

#endif
                                                                                                                                                                                                                                                                                                                                                     xenstore/main_c.c                                                                                   0000644 0001750 0001750 00000002533 11146404050 013655  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /*
    C main function for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>

#include <xenctrl.h>

#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/memory.h>
#include <caml/alloc.h>

int main(int argc, char *argv[], char *envp[])
{
    value *val;

    /* Wait before things might hang up */
    sleep(1);

    caml_startup(argv);
    val = caml_named_value("main");
    if (!val) {
        printf("Couldn't find Caml main");
        return 1;
    }

    caml_callback(*val, Val_int(0));

    return 0;
}
                                                                                                                                                                     xenstore/main.ml                                                                                    0000644 0001750 0001750 00000006715 11146404050 013547  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Main functions for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Handle event *)
let handle_event xenstored =
  let port = Eventchan.pending () in
  if port <> Constants.null_file_descr
  then (
    if port = xenstored#virq_port
    then (
      let domains = xenstored#domains#cleanup in
      if List.length domains > 0
      then (
        List.iter (fun domain -> if domain#is_dying then xenstored#remove_domain domain) domains;
        xenstored#watches#fire_watches "@releaseDomain" false false
      )
    );
    if Eventchan.unmask port = - 1 then Utils.barf_perror "Failed to write to event channel"
  )
  else Utils.barf_perror ("Failed to read from event channel")

(* Handle I/O for domains *)
let handle_io xenstored =
  let handle_io_for_domain domain =
    try
      if domain#can_read then domain#read;
      if domain#has_input_message
      then (
        Trace.io domain#id "IN" (Os.get_time ()) (List.hd domain#input_messages);
        let msg_type = Message.message_type_to_string (List.hd domain#input_messages).Message.header.Message.message_type
        and msg_length = (List.hd domain#input_messages).Message.header.Message.length in
        if xenstored#options.Option.verbose then (Printf.printf "Got message %s len %d from %d\n" msg_type msg_length domain#id; flush stdout);
        Process.process xenstored domain
      );
      while domain#can_write do
        let msg_type = Message.message_type_to_string (List.hd domain#output_messages).Message.header.Message.message_type
        and msg_payload = (List.hd domain#output_messages).Message.payload in
        if xenstored#options.Option.verbose then (Printf.printf "Writing msg %s (%s) out to %d\n" msg_type msg_payload domain#id; flush stdout);
        Trace.io domain#id "OUT" (Os.get_time ()) (List.hd domain#output_messages);
        domain#write
      done
    with Constants.Xs_error (Constants.EIO, _, _) -> (
          (try if not (Domain.is_unprivileged domain) then while domain#can_write do domain#write done with _ -> ());
          xenstored#remove_domain domain;
          if Domain.is_unprivileged domain then xenstored#watches#fire_watches "@releaseDomain" false false
        )
  in
  List.iter handle_io_for_domain xenstored#domains#domains

(* Main method *)
let main =
  let options = Option.parse () in
  Option.check_options options;
  
  let store = new Store.store in
  let xenstored = new Xenstored.xenstored options store in
  
  Os.init ();
  
  let event_chan = xenstored#initialise_domains in
  
  while true do
    Os.check_connections xenstored event_chan;
    if Os.check_event_chan event_chan then handle_event xenstored;
    handle_io xenstored
  done

(* Register callback for main function *)
let _ = Callback.register "main" main
                                                   xenstore/Makefile                                                                                   0000644 0001750 0001750 00000007157 11146404050 013732  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 XEN_ROOT=../..
include $(XEN_ROOT)/tools/Rules.mk

MAJOR = 3.0
MINOR = 0

CFLAGS += -Werror
CFLAGS += -I.
CFLAGS += $(CFLAGS_libxenctrl)


CAMLLIB = $(shell ocamlc -where)
DEF_CPPFLAGS += -I$(CAMLLIB)

OCAMLFIND=ocamlfind
OCAMLOPT=ocamlopt


INCLUDES := -I .
OCAML_LIBS := unix.cmxa
C_LIBS := $(LDFLAGS_libxenctrl) -lpthread -lc

OBJS := constants.cmx utils.cmx eventchan.cmx interface.cmx xenbus.cmx socket.cmx message.cmx connection.cmx dominfo.cmx trace.cmx store.cmx domain.cmx os.cmx option.cmx watch.cmx permission.cmx transaction.cmx xenstored.cmx process.cmx main.cmx
C_OBJS := xenbus_c.o eventchan_c.o dominfo_c.o main_c.o

ATTACK_OBJS := constants.cmx utils.cmx interface.cmx socket.cmx message.cmx connection.cmx store.cmx attack.cmx


# Build rules

.PHONY: all default clean

all: xenstored attack libxenstore.so libxenstore.a clients
default: all


# Source build rules

%.cmx: %.ml
	$(OCAMLFIND) $(OCAMLOPT) $(INCLUDES) -c $< -o $@

%.o: %.c
	$(CC) $(CFLAGS) -I$(CAMLLIB) -c $< -o $@


# Executable build rules

xenstored: $(OBJS) $(C_OBJS)
	$(OCAMLFIND) $(OCAMLOPT) -o xenstored $(OCAML_LIBS) $(OBJS) $(C_OBJS) -ccopt '$(CFLAGS)' -cclib '$(C_LIBS)' -cclib '$(LDFLAGS)'


attack: $(ATTACK_OBJS)
	$(OCAMLFIND) $(OCAMLOPT) unix.cmxa -o attack $(ATTACK_OBJS) -ccopt '$(CFLAGS)' -cclib '$(C_LIBS)' -cclib '$(LDFLAGS)'


CLIENTS := xenstore-exists xenstore-list xenstore-read xenstore-rm xenstore-chmod
CLIENTS += xenstore-write xenstore-ls

XENSTORED_OBJS = xs_lib.o

ifneq ($(XENSTORE_STATIC_CLIENTS),y)
LIBXENSTORE := libxenstore.so
else
LIBXENSTORE := libxenstore.a
xenstore xenstore-control: CFLAGS += -static
endif


.PHONY: clients
clients: xenstore $(CLIENTS) xenstore-control


$(CLIENTS): xenstore
	ln -f xenstore $@

xenstore: xenstore_client.o $(LIBXENSTORE)
	$(CC) $(CFLAGS) $(LDFLAGS) $< -L. -lxenstore $(SOCKET_LIBS) -o $@

xenstore-control: xenstore_control.o $(LIBXENSTORE)
	$(CC) $(CFLAGS) $(LDFLAGS) $< -L. -lxenstore $(SOCKET_LIBS) -o $@


libxenstore.so: libxenstore.so.$(MAJOR)
	ln -sf $< $@
libxenstore.so.$(MAJOR): libxenstore.so.$(MAJOR).$(MINOR)
	ln -sf $< $@

xs.opic: CFLAGS += -DUSE_PTHREAD

libxenstore.so.$(MAJOR).$(MINOR): xs.opic xs_lib.opic
	$(CC) $(CFLAGS) $(LDFLAGS) -Wl,$(SONAME_LDFLAG) -Wl,libxenstore.so.$(MAJOR) $(SHLIB_CFLAGS) -o $@ $^ $(SOCKET_LIBS) -lpthread

libxenstore.a: xs.o xs_lib.o
	$(AR) rcs $@ $^


# Cleaning rules

.PHONY: clean
clean: clean-xenstored clean attack clean-xs

clean-xenstored:
	rm -f *.a *.o *.cmx *.cmi xenstored

clean-attack:
	rm -f *.a *.o *.cmx *.cmi attack

clean-xs:
	rm -f *.a *.o *.opic *.so* xenstored_probes.h
	rm -f xs_random xs_stress xs_crashme
	rm -f xenstore-control
	rm -f xenstore $(CLIENTS)
	$(RM) $(DEP)


.PHONY: TAGS
TAGS:
	etags `find . -name '*.[ch]'`

.PHONY: tarball
tarball: clean
	cd .. && tar -c -j -v -h -f xenstore.tar.bz2 xenstore/


# Install rules

.PHONY: install
install: all
	$(INSTALL_DIR) $(DESTDIR)$(BINDIR)
	$(INSTALL_DIR) $(DESTDIR)$(SBINDIR)
	$(INSTALL_PROG) xenstored $(DESTDIR)$(SBINDIR)
	$(INSTALL_PROG) xenstore-control $(DESTDIR)$(BINDIR)
	$(INSTALL_PROG) xenstore $(DESTDIR)/usr/bin
	set -e ; for c in $(CLIENTS) ; do \
		ln -f $(DESTDIR)/usr/bin/xenstore $(DESTDIR)/usr/bin/$${c} ; \
	done
	$(INSTALL_DIR) $(DESTDIR)$(LIBDIR)
	$(INSTALL_PROG) libxenstore.so.$(MAJOR).$(MINOR) $(DESTDIR)$(LIBDIR)
	ln -sf libxenstore.so.$(MAJOR).$(MINOR) $(DESTDIR)$(LIBDIR)/libxenstore.so.$(MAJOR)
	ln -sf libxenstore.so.$(MAJOR) $(DESTDIR)$(LIBDIR)/libxenstore.so
	$(INSTALL_DATA) libxenstore.a $(DESTDIR)$(LIBDIR)
	$(INSTALL_DATA) xs.h $(DESTDIR)$(INCLUDEDIR)
	$(INSTALL_DATA) xs_lib.h $(DESTDIR)$(INCLUDEDIR)


-include $(DEPS)

# never delete any intermediate files.
.SECONDARY:
                                                                                                                                                                                                                                                                                                                                                                                                                 xenstore/message.ml                                                                                 0000644 0001750 0001750 00000012072 11146404050 014240  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Messages for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let header_size = 16;

(* XenStore message types *)
type xs_message_type =
  | XS_DEBUG
  | XS_DIRECTORY
  | XS_READ
  | XS_GET_PERMS
  | XS_WATCH
  | XS_UNWATCH
  | XS_TRANSACTION_START
  | XS_TRANSACTION_END
  | XS_INTRODUCE
  | XS_RELEASE
  | XS_GET_DOMAIN_PATH
  | XS_WRITE
  | XS_MKDIR
  | XS_RM
  | XS_SET_PERMS
  | XS_WATCH_EVENT
  | XS_ERROR
  | XS_IS_DOMAIN_INTRODUCED
  | XS_RESUME
  | XS_SET_TARGET
  | XS_UNKNOWN

(* Convert a message type to an int32 *)
let xs_message_type_to_int32 message_type =
  match message_type with
  | XS_DEBUG -> 0l
  | XS_DIRECTORY -> 1l
  | XS_READ -> 2l
  | XS_GET_PERMS -> 3l
  | XS_WATCH -> 4l
  | XS_UNWATCH -> 5l
  | XS_TRANSACTION_START -> 6l
  | XS_TRANSACTION_END -> 7l
  | XS_INTRODUCE -> 8l
  | XS_RELEASE -> 9l
  | XS_GET_DOMAIN_PATH -> 10l
  | XS_WRITE -> 11l
  | XS_MKDIR -> 12l
  | XS_RM -> 13l
  | XS_SET_PERMS -> 14l
  | XS_WATCH_EVENT -> 15l
  | XS_ERROR -> 16l
  | XS_IS_DOMAIN_INTRODUCED -> 17l
  | XS_RESUME -> 18l
  | XS_SET_TARGET -> 19l
  | XS_UNKNOWN -> - 1l

(* Convert an int32 to a message type *)
let int32_to_message_type xs_message_type =
  match xs_message_type with
  | 0l -> XS_DEBUG
  | 1l -> XS_DIRECTORY
  | 2l -> XS_READ
  | 3l -> XS_GET_PERMS
  | 4l -> XS_WATCH
  | 5l -> XS_UNWATCH
  | 6l -> XS_TRANSACTION_START
  | 7l -> XS_TRANSACTION_END
  | 8l -> XS_INTRODUCE
  | 9l -> XS_RELEASE
  | 10l -> XS_GET_DOMAIN_PATH
  | 11l -> XS_WRITE
  | 12l -> XS_MKDIR
  | 13l -> XS_RM
  | 14l -> XS_SET_PERMS
  | 15l -> XS_WATCH_EVENT
  | 16l -> XS_ERROR
  | 17l -> XS_IS_DOMAIN_INTRODUCED
  | 18l -> XS_RESUME
  | 19l -> XS_SET_TARGET
  | _ -> XS_UNKNOWN

(* Return string representation of a message type *)
let message_type_to_string message_type =
  match message_type with
  | XS_DEBUG -> "DEBUG"
  | XS_DIRECTORY -> "DIRECTORY"
  | XS_READ -> "READ"
  | XS_GET_PERMS -> "GET_PERMS"
  | XS_WATCH -> "WATCH"
  | XS_UNWATCH -> "UNWATCH"
  | XS_TRANSACTION_START -> "TRANSACTION_START"
  | XS_TRANSACTION_END -> "TRANSACTION_END"
  | XS_INTRODUCE -> "INTRODUCE"
  | XS_RELEASE -> "RELEASE"
  | XS_GET_DOMAIN_PATH -> "GET_DOMAIN_PATH"
  | XS_WRITE -> "WRITE"
  | XS_MKDIR -> "MKDIR"
  | XS_RM -> "RM"
  | XS_SET_PERMS -> "SET_PERMS"
  | XS_WATCH_EVENT -> "WATCH_EVENT"
  | XS_ERROR -> "ERROR"
  | XS_IS_DOMAIN_INTRODUCED -> "IS_DOMAIN_INTRODUCED"
  | XS_RESUME -> "RESUME"
  | XS_SET_TARGET -> "SET_TARGET"
  | XS_UNKNOWN -> "UNKNOWN"

(* Message header *)
type header =
  {
    message_type : xs_message_type;
    transaction_id : int32;
    request_id : int32;
    length : int
  }

(* Message *)
type message =
  {
    header : header;
    payload : string
  }

(* Make a message *)
let make message_type transaction_id request_id payload =
  {
    header =
      {
        message_type = message_type;
        transaction_id = transaction_id;
        request_id = request_id;
        length = (String.length payload)
      };
    payload = payload
  }

(* Null message *)
let null_message = make XS_UNKNOWN 0l 0l Constants.null_string

(* ACK message *)
let ack message =
  make message.header.message_type message.header.transaction_id message.header.request_id (Utils.null_terminate "OK")

(* Error message *)
let error message error =
  make XS_ERROR message.header.transaction_id message.header.request_id (Utils.null_terminate (Constants.error_message error))

(* Event message *)
let event payload =
  make XS_WATCH_EVENT 0l 0l payload

(* Reply message *)
let reply message payload =
  make message.header.message_type message.header.transaction_id message.header.request_id payload

(* Deserialise a message header from a string *)(* Null message *)
let deserialise_header buffer =
  {
    message_type = int32_to_message_type (Utils.bytes_to_int32 (String.sub buffer 0 4));
    transaction_id = Utils.bytes_to_int32 (String.sub buffer 8 4);
    request_id = Utils.bytes_to_int32 (String.sub buffer 4 4);
    length = Utils.bytes_to_int (String.sub buffer 12 4)
  }

(* Serialise a message header to a string *)
let serialise_header header =
  let message_type = Utils.int32_to_bytes (xs_message_type_to_int32 header.message_type)
  and transaction_id = Utils.int32_to_bytes header.transaction_id
  and request_id = Utils.int32_to_bytes header.request_id
  and length = Utils.int_to_bytes header.length in
  message_type ^ request_id ^ transaction_id ^ length
                                                                                                                                                                                                                                                                                                                                                                                                                                                                      xenstore/option.ml                                                                                  0000644 0001750 0001750 00000010047 11146404050 014124  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Options for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Options type *)
type t = {
  fork : bool;
  output_pid : bool;
  domain_init : bool;
  separate_domain : bool;
  pid_file : string;
  trace_file : string;
  recovery : bool;
  verbose : bool;
  quota_num_entries_per_domain : int;
  quota_max_entry_size : int;
  quota_num_watches_per_domain : int;
  quota_max_transaction : int
}

(* Usage message header *)
let usage = "Usage:\n  xenstored <options>\n\nwhere options may include:\n"

(* Parse command-line options *)
let parse () =
  (* Default options *)
  let fork = ref true
  and output_pid = ref false
  and domain_init = ref true
  and pid_file = ref ""
  and trace_file = ref ""
  and recovery = ref true
  and verbose = ref false
  and separate_domain = ref false
  and quota_num_entries_per_domain = ref 1000
  and quota_max_entry_size = ref 2048
  and quota_num_watches_per_domain = ref 128
  and quota_max_transaction = ref 10 in
  
  (* Command-line arguments list *)
  let spec_list = Arg.align [
      ("--no-domain-init", Arg.Clear domain_init, " to state that xenstored should not initialise dom0,");
      ("--pid-file", Arg.Set_string pid_file, "<file> giving a file for the daemon's pid to be written,");
      ("--no-fork", Arg.Clear fork, " to request that the daemon does not fork,");
      ("--output-pid", Arg.Set output_pid, " to request that the pid of the daemon is output,");
      ("--trace-file", Arg.String (fun s -> trace_file := s; Trace.traceout := Some (open_out s)), "<file> giving the file for logging,");
      ("--entry-nb", Arg.Set_int quota_num_entries_per_domain, "<nb> limit the number of entries per domain,");
      ("--entry-size", Arg.Set_int quota_max_entry_size, "<size> limit the size of entry per domain,");
      ("--entry-watch", Arg.Set_int quota_num_watches_per_domain,"<nb> limit the number of watches per domain,");
      ("--transaction", Arg.Set_int quota_max_transaction, "<nb> limit the number of transaction allowed per domain,");
      ("--no-recovery", Arg.Clear recovery, " to request that no recovery should be attempted when the store is corrupted (debug only),");
      ("--preserve-local", Arg.Unit (fun () -> ()), " to request that /local is preserved on start-up,");
      ("--verbose", Arg.Set verbose, " to request verbose execution.");
      ("--separate-dom", Arg.Set separate_domain, " xenstored runs in it's own domain.");
      ] in
  
  (* Parse command-line arguments *)
  Arg.parse spec_list Os.parse_option usage;
  
  (* Set and return chosen options *)
  {
    fork = !fork;
    output_pid = !output_pid;
    domain_init = !domain_init;
    separate_domain = !separate_domain;
    pid_file = !pid_file;
    trace_file = !trace_file;
    recovery = !recovery;
    verbose = !verbose;
    quota_num_entries_per_domain = !quota_num_entries_per_domain;
    quota_max_entry_size = !quota_max_entry_size;
    quota_num_watches_per_domain = !quota_num_watches_per_domain;
    quota_max_transaction = !quota_max_transaction
  }

let check_options options =
  if not options.domain_init && options.separate_domain then Utils.barf_perror "Incompatible options";
  if options.fork then Os.daemonise ();
  if options.pid_file <> Constants.null_string then Os.write_pid_file options.pid_file;
  if options.output_pid then (Printf.printf "%d\n" (Os.get_pid ()); flush stdout)
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                         xenstore/os.ml                                                                                      0000644 0001750 0001750 00000017321 11146404050 013237  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    OS-specific code for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let xenstored_proc_domid = "/proc/xen/xsd_domid"
let xenstored_proc_dom0_port = "/proc/xen/xsd_dom0_port"
let xenstored_proc_dom0_mfn = "/proc/xen/xsd_dom0_mfn"
let xenstored_proc_kva = "/proc/xen/xsd_kva"
let xenstored_proc_port = "/proc/xen/xsd_port"

(* Change the permissions for a socket address *)
let xsd_chmod addr =
  match addr with
  | Unix.ADDR_UNIX name -> Unix.chmod name 0o600
  | _ -> Utils.barf_perror "addr -- chmod oops"

(* Get a XenStore daemon directory *)
let xsd_getdir env_var fallback =
  try Sys.getenv env_var with Not_found -> fallback

(* Create the given XenStore daemon directory, if needed *)
let xsd_mkdir name =
  if not (Sys.file_exists name) then Unix.mkdir name 0o755

(* Return the XenStore daemon run directory *)
let xsd_rundir () =
  xsd_getdir "XENSTORED_RUNDIR" "/var/run/xenstored"

(* Return the XenStore daemon path *)
let xsd_socket_path () =
  xsd_getdir "XENSTORED_PATH" ((xsd_rundir ()) ^ "/socket")

(* Return the name of the XenStore daemon read-only socket *)
let xsd_socket_ro () =
  (xsd_socket_path ()) ^ "_ro"

(* Return the name of the XenStore daemon read-write socket *)
let xsd_socket_rw () =
  xsd_socket_path ()

(* Remove the old sockets *)
let xsd_unlink addr =
  match addr with
  | Unix.ADDR_UNIX name -> if Sys.file_exists name then Unix.unlink name
  | _ -> Utils.barf_perror "addr -- unlink oops"

let conn_fds = Hashtbl.create 8
let conn_id = ref (- 1)
let in_set = ref []
let out_set = ref []

(* Accept a connection *)
let accept socket can_write in_set out_set =
  let (fd, _) = Unix.accept socket in
  let interface = new Socket.socket_interface fd can_write in_set out_set in
  let connection = new Connection.connection interface in
  let domu = new Domain.domain !conn_id connection in
  decr conn_id;
  Hashtbl.add conn_fds domu#id fd;
  domu

(* Create and listen to a socket *)
let create_socket socket_name =
  xsd_mkdir (xsd_rundir ());
  let addr = Unix.ADDR_UNIX socket_name
  and socket = Unix.socket Unix.PF_UNIX Unix.SOCK_STREAM 0 in
  xsd_unlink addr;
  Unix.bind socket addr;
  xsd_chmod addr;
  Unix.listen socket 1;
  socket

let filter_conn_fds conn_fds domains =
  let active_conn_ids = List.fold_left (fun ids domain -> if domain#id < 0 then domain#id :: ids else ids) [] domains in
  Hashtbl.iter (fun id fd -> if not (List.mem id active_conn_ids) then Hashtbl.remove conn_fds id) conn_fds

(* Fork daemon *)
let fork_daemon () =
  let pid = Unix.fork () in
  if pid < 0 then Utils.barf_perror ("Failed to fork daemon: " ^ (string_of_int pid));
  if pid <> 0 then exit 0

(* Return the (input) socket connections *)
let get_input_socket_connections conn_fds =
  Hashtbl.fold (fun _ fd rest -> fd :: rest) conn_fds []

(* Return the (output) socket connections *)
let get_output_socket_connections domains conn_fds =
  List.fold_left (fun rest domain -> if domain#can_write then Hashtbl.find conn_fds domain#id :: rest else rest) [] (List.filter (fun domain -> Hashtbl.mem conn_fds domain#id) domains)

(* Read a value from a proc file *)
let read_int_from_proc name =
  let fd = Unix.openfile name [ Unix.O_RDONLY ] 0o600
  and buff = String.create 20 in
  let int = Unix.read fd buff 0 (String.length buff) in
  Unix.close fd;
  if int <> Constants.null_file_descr then int_of_string (String.sub buff 0 int) else Constants.null_file_descr

let socket_rw = create_socket (xsd_socket_rw ())
let socket_ro = create_socket (xsd_socket_ro ())
let special_fds = ref [ socket_rw; socket_ro ]

(* Check connections *)
let check_connections xenstored event_chan =
  filter_conn_fds conn_fds xenstored#domains#domains;
  
  let input_conns = get_input_socket_connections conn_fds
  and output_conns = get_output_socket_connections xenstored#domains#domains conn_fds
  and timeout = xenstored#domains#timeout in
  
  let (i_set, o_set, _) = Unix.select ((if event_chan <> Constants.null_file_descr then Socket.file_descr_of_int event_chan :: !special_fds else !special_fds) @ input_conns) output_conns [] timeout in
  in_set := i_set;
  out_set := o_set;
  
  if List.mem socket_rw !in_set then xenstored#add_domain (accept socket_rw true in_set out_set);
  if List.mem socket_ro !in_set then xenstored#add_domain (accept socket_ro false in_set out_set)

(* Check the event channel for an event *)
let check_event_chan event_chan =
  List.mem (Socket.file_descr_of_int event_chan) !in_set

(* Daemonise *)
let daemonise () =
  (* Separate from parent via fork, so init inherits us *)
  fork_daemon ();
  
  (* Session leader so ^C doesn't whack us *)
  ignore (Unix.setsid ());
  
  (* Let session leader exit so child cannot regain CTTY *)
  fork_daemon ();
  
  (* Move off any mount points we might be in *)
  (try Unix.chdir "/" with _ -> Utils.barf_perror "Failed to chdir");
  
  (* Discard parent's old-fashioned umask prejudices *)
  ignore (Unix.umask 0);
  
  (* Redirect outputs to null device *)
  let dev_null = Unix.openfile "/dev/null" [ Unix.O_RDWR ] 0o600 in
  Unix.dup2 dev_null Unix.stdin;
  Unix.dup2 dev_null Unix.stdout;
  Unix.dup2 dev_null Unix.stderr;
  Unix.close dev_null

(* Return the XenStore domain ID *)
let get_domxs_id () =
  read_int_from_proc xenstored_proc_domid

(* Return the Domain-0 mfn *)
let get_dom0_mfn () =
  read_int_from_proc xenstored_proc_dom0_mfn

(* Return the Domain-0 port *)
let get_dom0_port () =
  read_int_from_proc xenstored_proc_dom0_port

(* Return the pid *)
let get_pid () =
  Unix.getpid ()

(* Return the current time *)
let get_time () =
  let tm = Unix.localtime (Unix.gettimeofday ()) in
  let year = tm.Unix.tm_year + 1900
  and month = tm.Unix.tm_mon + 1
  and day = tm.Unix.tm_mday
  and hour = tm.Unix.tm_hour
  and minute = tm.Unix.tm_min
  and second = tm.Unix.tm_sec in
  Printf.sprintf "%04d%02d%02d %02d:%02d:%02d" year month day hour minute second;;

(* Return the XenBus port *)
let get_xenbus_port () =
  let fd = Unix.openfile xenstored_proc_port [ Unix.O_RDONLY ] 0
  and str = String.create 20 in
  let len = Unix.read fd str 0 (String.length str) in
  Unix.close fd;
  if len <> - 1 then int_of_string (String.sub str 0 len) else Constants.null_file_descr

(* OS specific initialisation *)
let init () =
  ignore (Sys.signal Sys.sigpipe Sys.Signal_ignore)

(* Map XenBus page *)
let map_xenbus port =
  let fd = Unix.openfile xenstored_proc_kva [ Unix.O_RDWR ] 0o600 in
  let interface = new Xenbus.xenbus_interface port (Xenbus.mmap (Socket.int_of_file_descr fd)) in
  Unix.close fd;
  interface

(* Extra option parsing, if needed *)
let parse_option option =
  ()

(* Write PID file *)
let write_pid_file pid_file =
  let fd = Unix.openfile pid_file [ Unix.O_RDWR; Unix.O_CREAT ] 0o600 in
  
  (* Exit silently if daemon already running *)
  (try Unix.lockf fd Unix.F_TLOCK 0 with _ -> ignore (exit 0));
  
  let pid = string_of_int (Unix.getpid ()) in
  let len = String.length pid in
  
  try
    if Unix.write fd pid 0 len <> len then Utils.barf_perror ("Writing pid file " ^ pid_file);
    Unix.close fd
  with _ -> Utils.barf_perror ("Writing pid file " ^ pid_file)
                                                                                                                                                                                                                                                                                                               xenstore/permission.ml                                                                              0000644 0001750 0001750 00000010516 11146404050 015005  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Permissions for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type access =
  | NONE
  | READ
  | WRITE
  | BOTH

type t =
  {
    access : access;
    domain_id : int
  }

let make access domain_id =
  {
    access = access;
    domain_id = domain_id
  }

let permission_of_string string =
  {
    access =
      (match string.[0] with
        | 'n' -> NONE
        | 'r' -> READ
        | 'w' -> WRITE
        | 'b' -> BOTH
        | _ -> raise (Constants.Xs_error (Constants.EINVAL, "permission_of_string", string)));
    domain_id = int_of_string (String.sub string 1 (pred (String.length string)))
  }

let string_of_permission permission =
  let perm_str =
    match permission.access with
    | NONE -> "n"
    | READ -> "r"
    | WRITE -> "w"
    | BOTH -> "b" in
  perm_str ^ (string_of_int permission.domain_id)

let check_access access1 access2 =
  match access1 with
  | READ | WRITE -> access2 = access1 || access2 = BOTH
  | _ -> access2 = access1

let compare permission1 permission2 =
  permission1.access = permission2.access && permission1.domain_id = permission2.domain_id

let get_path path =
  Store.root_path ^ ".permissions" ^ (if path = Store.root_path then Constants.null_string else path)

class permissions =
object(self)
  method add (store : string Store.store) (path : string) (domain_id : int) =
    let domain_id = if domain_id < 0 then 0 else domain_id
    and parent_path = Store.parent_path path in
    if not (store#node_exists (get_path parent_path)) then self#add store parent_path domain_id;
    let parent_permissions = self#get store parent_path in
    let new_permissions = if domain_id = 0 then parent_permissions else make (List.hd parent_permissions).access domain_id :: List.tl parent_permissions in
    self#set (List.map string_of_permission new_permissions) store path
  method check (store : string Store.store) path access domain_id =
    let domain_id = if domain_id < 0 then 0 else domain_id
    and permissions = self#get store path in
    if domain_id = 0
    then true
    else
      let default_permission = List.hd permissions
      and actual_permissions = List.tl permissions in
      if default_permission.domain_id = domain_id
      then true
      else check_access access (try (List.find (fun perm -> perm.domain_id = domain_id) actual_permissions).access with Not_found -> default_permission.access)
  method get (store : string Store.store) (path : string) =
    let ppath = get_path path in
    match store#read_node ppath with
    | Store.Value permissions | Store.Hack (permissions, _) -> List.map permission_of_string (Utils.split permissions)
    | Store.Empty -> raise (Constants.Xs_error (Constants.EINVAL, "Permission.permissions#get", ppath))
    | Store.Children _ ->
        let parent_path = Store.parent_path path in
        let parent_permissions = self#get store parent_path in
        self#set (List.map string_of_permission parent_permissions) store path;
        parent_permissions
  method remove (store : string Store.store) path = store#remove_node (get_path path)
  method set (permissions : string list) (store : string Store.store) (path : string) =
    let ppath = get_path path in
    let parent_path = Store.parent_path path in
    if not (path = Store.root_path) && not (store#node_exists (get_path parent_path))
    then (
      let domain_id = (permission_of_string (List.hd permissions)).domain_id in
      self#add store parent_path domain_id
    );
    ignore (try store#read_node ppath with _ -> store#create_node ppath; store#read_node ppath);
    store#write_node ppath (Utils.combine_with_string permissions (String.make 1 Constants.null_char));
end
                                                                                                                                                                                  xenstore/process.ml                                                                                 0000644 0001750 0001750 00000047037 11146404050 014303  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Processing for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Check for a valid domain ID *)
let check_domain_id domain_id =
  try int_of_string domain_id >= 0 with _ -> false

(* Check for a valid domain ID (only parameter) *)
let check_domain_id_only payload =
  let domain_id = List.hd (Utils.split payload) in
  String.length domain_id = pred (String.length payload) && check_domain_id domain_id

(* Check for 32-bit integer *)
let check_int int =
  try ignore (Int32.of_string int); true with _ -> false

(* Check introduce *)
let check_introduce payload =
  let split = Utils.split payload in
  let length = List.length split in
  (length = 3 || length = 4) && check_domain_id (List.nth split 0) && check_int (List.nth split 1) && check_int (List.nth split 2)

let rec check_chars path i =
  if i >= String.length path
  then true
  else if not (String.contains Store.valid_characters path.[i])
  then false
  else check_chars path (succ i)

(* Check for a valid path *)
let check_path path =
  if String.length path > 0
  then
    if path.[pred (String.length path)] <> Store.dividor
    then
      if not (Utils.strstr path "//")
      then
        if Store.is_relative path
        then
          if String.length path <= Constants.relative_path_max
          then check_chars path 0
          else false
        else if String.sub path 0 (String.length Store.root_path) = Store.root_path
        then
          if String.length path <= Constants.absolute_path_max
          then check_chars path 0
          else false
        else false
      else false
    else if path = Store.root_path then true else false
  else false

(* Check for a valid path (only parameter) *)
let check_path_only payload =
  let path = Utils.strip_null payload in
  succ (String.length path) = String.length payload && check_path path

let check_permissions payload =
  let split = Utils.split payload in
  let min_length = if payload.[pred (String.length payload)] = Constants.null_char then 2 else 3
  and perm_list = if payload.[pred (String.length payload)] = Constants.null_char then List.tl split else Utils.remove_last (List.tl split) in
  List.length split >= min_length && check_path (List.nth split 0) && List.fold_left (fun accum perm -> accum && (try ignore (Permission.permission_of_string perm); true with _ -> false)) true perm_list

(* Check for a valid transaction end *)
let check_transaction_end payload =
  let value = Utils.strip_null payload in
  succ (String.length value) = String.length payload && (value = Constants.payload_true || value = Constants.payload_false)

(* Check for a valid transaction start *)
let check_transaction_start payload =
  String.length payload = 1 && payload.[0] = Constants.null_char

(* Check for a valid watch path *)
let check_watch_path path =
  if Store.is_event path then check_chars path 0 else check_path path

(* TODO: Check for a valid watch token *)
let check_watch_token token =
  true

(* Check for a valid watch/unwatch *)
let check_watch payload =
  let split = Utils.split payload in
  let length = List.length split in
  (length = 2 || length = 3) && check_watch_path (List.nth split 0) && check_watch_token (List.nth split 1)

let check_write payload =
  let split = Utils.split payload in
  let length = List.length split in
  (length = 1 || length = 2) && check_path (List.nth split 0)

(* Check a message to make sure the payload is valid *)
let check message =
  match message.Message.header.Message.message_type with
  | Message.XS_DIRECTORY -> check_path_only message.Message.payload
  | Message.XS_GET_DOMAIN_PATH -> check_path_only message.Message.payload
  | Message.XS_GET_PERMS -> check_path_only message.Message.payload
  | Message.XS_INTRODUCE -> check_introduce message.Message.payload
  | Message.XS_IS_DOMAIN_INTRODUCED -> check_path_only message.Message.payload
  | Message.XS_MKDIR -> check_path_only message.Message.payload
  | Message.XS_READ -> check_path_only message.Message.payload
  | Message.XS_RELEASE -> check_path_only message.Message.payload
  | Message.XS_RESUME -> check_path_only message.Message.payload
  | Message.XS_RM -> check_path_only message.Message.payload
  | Message.XS_SET_PERMS -> check_permissions message.Message.payload
  | Message.XS_TRANSACTION_END -> check_transaction_end message.Message.payload
  | Message.XS_TRANSACTION_START -> check_transaction_start message.Message.payload
  | Message.XS_UNWATCH -> check_watch message.Message.payload
  | Message.XS_WATCH -> check_watch message.Message.payload
  | Message.XS_WRITE -> check_write message.Message.payload
  | _ -> false

(* Return the list of parent paths that will be created for a given path *)
let rec created_paths store path =
  if store#node_exists path then [] else path :: created_paths store (Store.parent_path path)

(* Return the list of child paths that will be deleted for a given path *)
let rec removed_paths store path =
  match store#read_node path with
  | Store.Children children | Store.Hack (_, children) -> List.fold_left (fun paths child -> paths @ (removed_paths store child#path)) [] children
  | _ -> [ path ]

(* Process a directory message *)
let process_directory domain store xenstored message =
  let path = Store.canonicalise domain (Utils.strip_null message.Message.payload) in
  try
    if xenstored#permissions#check store path Permission.READ domain#id
    then
      let payload =
        match store#read_node path with
        | Store.Children (children) | Store.Hack (_, children) -> List.fold_left (fun children_string child -> if check_path child#path then children_string ^ (Utils.null_terminate (Store.base_path child#path)) else children_string) Constants.null_string children
        | _ -> Constants.null_string in
      domain#add_output_message (Message.reply message payload)
    else domain#add_output_message (Message.error message Constants.EACCES)
  with Constants.Xs_error (errno, _, _) -> domain#add_output_message (Message.error message errno)

(* Process a get domain path message *)
let process_get_domain_path domain store message =
  let domid = Utils.strip_null message.Message.payload in
  let path = Utils.null_terminate (Store.domain_root ^ domid) in
  domain#add_output_message (Message.reply message path)

(* Process a get permissions message *)
let process_get_perms domain store xenstored message =
  let path = Store.canonicalise domain (Utils.strip_null message.Message.payload) in
  if xenstored#permissions#check store path Permission.READ domain#id
  then
    let permissions = xenstored#permissions#get store path in
    let payload = List.fold_left (fun permissions_string permission -> permissions_string ^ (Utils.null_terminate (Permission.string_of_permission permission))) Constants.null_string permissions in
    domain#add_output_message (Message.reply message payload)
  else domain#add_output_message (Message.error message Constants.EACCES)

(* Process an introduce message *)
let process_introduce domain store xenstored message =
  let split = Utils.split message.Message.payload in
  let domid = List.nth split 0
  and mfn = List.nth split 1
  and port = List.nth split 2
  and reserved = if List.length split = 4 then List.nth split 3 else Constants.null_string in
  if not (Domain.is_unprivileged domain)
  then (
    (* XXX: Reserved value *)
    if String.length reserved > 0 then ();
    let domu = Domain.domu_init (int_of_string domid) (int_of_string port) (int_of_string mfn) false in
    xenstored#add_domain domu;
    xenstored#watches#fire_watches "@introduceDomain" (message.Message.header.Message.transaction_id <> 0l) false;
    domain#add_output_message (Message.ack message)
  )
  else domain#add_output_message (Message.error message Constants.EACCES)

(* Process a is domains introduced message *)
let process_is_domain_introduced domain store xenstored message =
  let domid = int_of_string (Utils.strip_null message.Message.payload) in
  let domain_exists = try xenstored#domains#find_by_id domid; true with Not_found -> false in
  let payload = Utils.null_terminate (if domid = Constants.domain_id_self || domain_exists then Constants.payload_true else Constants.payload_false) in
  domain#add_output_message (Message.reply message payload)

(* Process a mkdir message *)
let process_mkdir domain store xenstored message =
  let path = Store.canonicalise domain (Utils.strip_null message.Message.payload)
  and transaction = Transaction.make domain#id message.Message.header.Message.transaction_id in
  (* If permissions exist, node already exists *)
  try
    if xenstored#permissions#check store path Permission.WRITE domain#id
    then domain#add_output_message (Message.ack message)
    else domain#add_output_message (Message.error message Constants.EACCES)
  with _ ->
      try
        if not (store#node_exists path)
        then (
          let paths = created_paths store path in
          store#create_node path;
          xenstored#permissions#add store path domain#id;
          List.iter (fun path -> xenstored#domain_entry_incr store transaction path) paths;
          if message.Message.header.Message.transaction_id = 0l
          then (
            xenstored#transactions#invalidate path;
            xenstored#watches#fire_watches path false false
          )
        );
        domain#add_output_message (Message.ack message)
      with e -> raise e (*domain#add_output_message (Message.error message Constants.EINVAL)*)

(* Process a read message *)
let process_read domain store xenstored message =
  let path = Store.canonicalise domain (Utils.strip_null message.Message.payload) in
  try
    if xenstored#permissions#check store path Permission.READ domain#id
    then
      let payload =
        match store#read_node path with
        | Store.Value value | Store.Hack (value, _) -> value
        | _ -> Constants.null_string in
      domain#add_output_message (Message.reply message payload)
    else domain#add_output_message (Message.error message Constants.EACCES)
  with Constants.Xs_error (errno, _, _) -> domain#add_output_message (Message.error message errno)

(* Process a release message *)
let process_release domain store xenstored message =
  if domain#id <= 0
  then
    let domu_id = int_of_string (Utils.strip_null message.Message.payload) in
    try
      xenstored#remove_domain (xenstored#domains#find_by_id domu_id);
      if domu_id > 0 then xenstored#watches#fire_watches "@releaseDomain" false false;
      domain#add_output_message (Message.ack message)
    with Not_found -> domain#add_output_message (Message.error message Constants.ENOENT)
  else domain#add_output_message (Message.error message Constants.EACCES)

(* Process a rm message *)
let process_rm domain store xenstored message =
  let path = Store.canonicalise domain (Utils.strip_null message.Message.payload)
  and transaction = Transaction.make domain#id message.Message.header.Message.transaction_id in
  try
    if store#node_exists path
    then
      if xenstored#permissions#check store path Permission.WRITE domain#id
      then
        if path <> Store.root_path
        then (
          let paths = removed_paths store path in
          List.iter (fun path -> xenstored#domain_entry_decr store transaction path) paths;
          store#remove_node path;
          xenstored#permissions#remove store path;
          if message.Message.header.Message.transaction_id = 0l
          then (
            xenstored#transactions#invalidate path;
            xenstored#watches#fire_watches path false true
          );
          domain#add_output_message (Message.ack message)
        )
        else domain#add_output_message (Message.error message Constants.EINVAL)
      else domain#add_output_message (Message.error message Constants.EACCES)
    else if store#node_exists (Store.parent_path path)
    then
      if xenstored#permissions#check store (Store.parent_path path) Permission.WRITE domain#id
      then domain#add_output_message (Message.ack message)
      else domain#add_output_message (Message.error message Constants.EACCES)
    else domain#add_output_message (Message.error message Constants.ENOENT) (* XXX: This might be wrong *)
  with Constants.Xs_error (errno, _, _) -> domain#add_output_message (Message.error message errno)

(* Process a set permissions message *)
let process_set_perms domain store xenstored message =
  let split = Utils.split message.Message.payload in
  let path = Store.canonicalise domain (List.hd split) in
  let (permissions, reserved) =
    if message.Message.payload.[pred (String.length message.Message.payload)] = Constants.null_char
    then (List.tl split, Constants.null_string)
    else (Utils.remove_last (List.tl split), List.nth split (pred (List.length split))) in
  if xenstored#permissions#check store path Permission.WRITE domain#id
  then (
    (* XXX: Reserved value *)
    if String.length reserved > 0 then ();
    try
      xenstored#permissions#set permissions store path;
      xenstored#watches#fire_watches path (message.Message.header.Message.transaction_id <> 0l) false;
      domain#add_output_message (Message.ack message)
    with _ -> domain#add_output_message (Message.error message Constants.EACCES) (* XXX: errno? *)
  )
  else domain#add_output_message (Message.error message Constants.EACCES)

(* Process a transaction end message *)
let process_transaction_end domain store xenstored message =
  let transaction = Transaction.make domain#id message.Message.header.Message.transaction_id in
  if xenstored#transactions#exists transaction
  then (
    Trace.destroy domain#id "transaction";
    if Utils.strip_null message.Message.payload = Constants.payload_true
    then
      if xenstored#commit transaction
      then domain#add_output_message (Message.ack message)
      else domain#add_output_message (Message.error message Constants.EAGAIN)
    else domain#add_output_message (Message.ack message)
  )
  else domain#add_output_message (Message.error message Constants.ENOENT)

(* Process a transaction start message *)
let process_transaction_start domain store xenstored message =
  try
    if message.Message.header.Message.transaction_id = 0l
    then
      let transaction = xenstored#new_transaction domain store in
      let payload = Utils.null_terminate (Int32.to_string transaction.Transaction.transaction_id) in
      domain#add_output_message (Message.reply message payload)
    else domain#add_output_message (Message.error message Constants.EBUSY)
  with Constants.Xs_error (errno, _, _) -> domain#add_output_message (Message.error message errno)

(* Process an unwatch message *)
let process_unwatch domain store xenstored message =
  let split = Utils.split message.Message.payload in
  let path = List.nth split 0
  and token = List.nth split 1
  and reserved = if List.length split = 3 then List.nth split 2 else Constants.null_string in
  let relative = Store.is_relative path in
  let actual_path = if relative then Store.canonicalise domain path else path in
  (* XXX: Reserved value *)
  if String.length reserved > 0 then ();
  if xenstored#watches#remove (Watch.make domain actual_path token relative)
  then (
    Trace.destroy domain#id "watch";
    domain#add_output_message (Message.ack message)
  )
  else domain#add_output_message (Message.error message Constants.ENOENT)

(* Process a watch message *)
let process_watch domain store xenstored message =
  let split = Utils.split message.Message.payload in
  let path = List.nth split 0
  and token = List.nth split 1
  and reserved = if List.length split = 3 then List.nth split 2 else Constants.null_string in
  let relative = Store.is_relative path in
  let actual_path = if relative then Store.canonicalise domain path else path in
  (* XXX: Reserved value *)
  if String.length reserved > 0 then ();
  if xenstored#add_watch domain (Watch.make domain actual_path token relative)
  then (
    Trace.create domain#id "watch";
    domain#add_output_message (Message.ack message);
    domain#add_output_message (Message.event ((Utils.null_terminate path) ^ (Utils.null_terminate token)))
  )
  else domain#add_output_message (Message.error message Constants.EEXIST)

(* Process a write message *)
let process_write domain store xenstored message =
  let split = Utils.split message.Message.payload in
  let path = Store.canonicalise domain (List.hd split)
  and value = Utils.combine (List.tl split) in
  let transaction = Transaction.make domain#id message.Message.header.Message.transaction_id in
  if not (store#node_exists path) || xenstored#permissions#check store path Permission.WRITE domain#id
  then
    if Domain.is_unprivileged domain && String.length value >= xenstored#options.Option.quota_max_entry_size
    then domain#add_output_message (Message.error message Constants.ENOSPC)
    else
      try
        if not (store#node_exists path)
        then (
          let paths = created_paths store path in
          store#create_node path;
          xenstored#permissions#add store path domain#id;
          List.iter (fun path -> xenstored#domain_entry_incr store transaction path) paths
        );
        store#write_node path value;
        if message.Message.header.Message.transaction_id = 0l
        then (
          xenstored#transactions#invalidate path;
          xenstored#watches#fire_watches path false false
        );
        domain#add_output_message (Message.ack message)
      with e -> raise e (*domain#add_output_message (Message.error message Constants.EINVAL)*) (* XXX: Wrong error? *)
  else domain#add_output_message (Message.error message Constants.EACCES)

(* Process a message *)
let process (xenstored : Xenstored.xenstored) domain =
  let message = domain#input_message in
  let store = xenstored#transactions#store (Transaction.make domain#id message.Message.header.Message.transaction_id) in
  if check message
  then (
    match message.Message.header.Message.message_type with
    | Message.XS_DIRECTORY -> process_directory domain store xenstored message
    | Message.XS_GET_DOMAIN_PATH -> process_get_domain_path domain store message
    | Message.XS_GET_PERMS -> process_get_perms domain store xenstored message
    | Message.XS_INTRODUCE -> process_introduce domain store xenstored message
    | Message.XS_IS_DOMAIN_INTRODUCED -> process_is_domain_introduced domain store xenstored message
    | Message.XS_MKDIR -> process_mkdir domain store xenstored message
    | Message.XS_READ -> process_read domain store xenstored message
    | Message.XS_RELEASE -> process_release domain store xenstored message
    | Message.XS_RM -> process_rm domain store xenstored message
    | Message.XS_SET_PERMS -> process_set_perms domain store xenstored message
    | Message.XS_TRANSACTION_END -> process_transaction_end domain store xenstored message
    | Message.XS_TRANSACTION_START -> process_transaction_start domain store xenstored message
    | Message.XS_UNWATCH -> process_unwatch domain store xenstored message
    | Message.XS_WATCH -> process_watch domain store xenstored message
    | Message.XS_WRITE -> process_write domain store xenstored message
    | _ -> domain#add_output_message (Message.error message Constants.EINVAL)
  )
  else domain#add_output_message (Message.error message Constants.EINVAL)
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 xenstore/README                                                                                     0000644 0001750 0001750 00000003113 11146404050 013136  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 OCaml XenStore


This is the second version of the OCaml XenStore daemon. It is functionally
equivalent to the C XenStore daemon, however certain message operations are
unimplemented. These are: DEBUG, RESUME, and SET_TARGET, which I suggested
in the new version of the XenStore protocol are unneeded anyway.

Due to some broken tools, a hack was added to support values in non-leaf nodes.
This can be found by the Hack type of Node in the Store. Ideally this would be
fixed so that there is no need for the hack.

The trace and verbose output has been changed slightly to show the domain ID
instead of a hex address. For socket connections, a negative domain ID is used.

Transactions have been improved to use optimistic concurrency control and
copy-on-write (instead of duplicating the entire store). A denial-of-service
attack has been included in the build. When run against the current version
of XenStore it will prevent any transaction from completing, thus effective
locking out XenStore. However, using the improved transaction implementation
in the OCaml XenStore, this attack no longer succeeds.


The development environment was 32-bit Ubuntu 8.10 with the stock OCaml package
version 3.10.2. It has been tested on Ubuntu 8.04 with the latest version of
xen-unstable and OCaml version 3.10.0.



To compile xenstored, the attack, and the libxenstore libraries simply type:

# make

To install, type:

# make install


The OCaml XenStore is a drop-in replacement the original C one and will be
compiled when Xen (or the tools) are built and will be installed on to the
system when Xen is installed.
                                                                                                                                                                                                                                                                                                                                                                                                                                                     xenstore/socket.ml                                                                                  0000644 0001750 0001750 00000003511 11146404050 014102  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Socket for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Convert an int to a file descriptor *)
external file_descr_of_int : int -> Unix.file_descr = "%identity"

(* Convert a file descriptor to an int *)
let int_of_file_descr fd = (Obj.magic (fd: Unix.file_descr) : int)

(* Socket interface *)
class socket_interface fd can_write in_set out_set =
object (self)
  inherit Interface.interface as super
  val m_fd = fd
  val m_can_write = can_write
  val m_in_set = in_set
  val m_out_set = out_set
  method private fd = m_fd
  method private in_set = !m_in_set
  method private out_set = !m_out_set
  method can_read = List.mem self#fd self#in_set
  method can_write = can_write
  method destroy = Unix.close self#fd
  method read buffer offset length =
    let bytes_read = Unix.read self#fd buffer offset length in
    if bytes_read = 0 && length <> 0
    then raise (Constants.Xs_error (Constants.EIO, "socket_interface#read", "could not read data"))
    else bytes_read
  method write buffer offset length = Unix.write self#fd buffer offset (min length (String.length buffer))
end
                                                                                                                                                                                       xenstore/store.ml                                                                                   0000644 0001750 0001750 00000013577 11146404050 013763  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Store for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* XenStore node contents type *)
type ('node, 'contents) node_contents =
  | Empty
  | Value of 'contents
  | Children of 'node list
  | Hack of 'contents * 'node list

let dividor = '/'
let dividor_str = String.make 1 dividor
let root_path = dividor_str
let domain_root = root_path ^ "local" ^ dividor_str ^ "domain" ^ dividor_str
let valid_characters = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-/_@"

(* Return the base path of a path *)
let base_path path =
  if path = root_path
  then path
  else
    let start = succ (String.rindex path dividor) in
    String.sub path start ((String.length path) - start)

(* Compare two nodes *)
let compare node1 node2 =
  String.compare node1#path node2#path

(* Check if a path is a child of another path *)
let is_child child parent =
  if parent = root_path
  then true
  else
    let length = min (String.length parent) (String.length child) in
    if String.sub child 0 length <> String.sub parent 0 length
    then false
    else
      let parent_length = String.length parent
      and child_length = String.length child in
      (* XXX: This returns child = parent *)
      if parent_length = child_length
      then true
      else if parent_length < child_length then String.get child parent_length = dividor
      else false

(* Check if a path is an event path *)
let is_event path =
  path.[0] = Constants.event_char

(* Check if a path is a relative path *)
let is_relative path =
  not (is_event path) && String.sub path 0 (String.length root_path) <> root_path

(* Iterate over nodes applying function f to each node *)
let rec iter f node =
  match node#contents with
  | Empty -> ()
  | Children children -> List.iter (fun child -> iter f child) children
  | Value value -> f value
  | Hack (value, children) -> f value; List.iter (fun child -> iter f child) children

(* Return the parent path of a path *)
let parent_path path =
  let slash = String.rindex path dividor in
  if slash = 0 then root_path else String.sub path 0 slash

(* Return canonicalised path *)
let canonicalise domain path =
  if not (is_relative path) then path else domain_root ^ (string_of_int domain#id) ^ dividor_str ^ path

(* XenStore node type *)
class ['contents] node path (contents : ('contents node, 'contents) node_contents) =
object (self)
  val m_path = path
  val mutable m_contents = contents
  method add_child child =
    match self#contents with
    | Empty -> m_contents <- Children [ child ]; true
    | Value value -> m_contents <- Hack (value, [ child ]); true (* false *)
    | Children children -> m_contents <- Children (List.sort compare (child :: children)); true
    | Hack (value, children) -> m_contents <- Hack (value, List.sort compare (child :: children)); true
  method contents = m_contents
  method path = m_path
  method get_child child_path =
    match self#contents with
    | Children children | Hack (_, children) -> (
          try List.find (fun child_node -> child_node#path = child_path) children
          with Not_found -> raise (Constants.Xs_error (Constants.ENOENT, "Store.node#get_child", child_path))
        )
    | _ -> raise (Constants.Xs_error (Constants.ENOENT, "Store.node#get_child", child_path))
  method remove_child child_path =
    match self#contents with
    | Children children -> m_contents <- Children (List.filter (fun child_node -> child_node#path <> child_path) children)
    | Hack (value, children) -> m_contents <- Hack (value, List.filter (fun child_node -> child_node#path <> child_path) children)
    | _ -> raise (Constants.Xs_error (Constants.ENOENT, "Store.node#remove_child", path))
  method set_contents contents = m_contents <- contents
end

class ['contents] store =
object (self)
  val m_root : 'contents node = new node root_path (Children [])
  method private construct_node path =
    let parent_path = parent_path path in
    let parent_node = try self#get_node parent_path with _ -> self#construct_node parent_path
    and node = new node path Empty in
    if parent_node#add_child node then node else raise (Constants.Xs_error (Constants.ENOENT, "Store.store#construct_node", path))
  method private get_node path = if path = root_path then self#root else (self#get_node (parent_path path))#get_child path
  method private root = m_root
  method create_node path = ignore (self#construct_node path)
  method iter f = iter f self#root
  method node_exists path = try ignore (self#get_node path); true with _ -> false
  method read_node path = (self#get_node path)#contents
  method remove_node path = (self#get_node (parent_path path))#remove_child path
  method replace_node (node : 'contents node) =
    let node_to_replace =
      if node#path = root_path
      then self#root
      else (
        if self#node_exists node#path then self#remove_node node#path;
        self#construct_node node#path
      ) in
    node_to_replace#set_contents node#contents
  method write_node path (contents : 'contents) =
    let node = self#get_node path in
    match node#contents with
    | Empty | Value _ -> node#set_contents (Value contents)
    | Children children | Hack (_, children) -> node#set_contents (Hack (contents, children))
end
                                                                                                                                 xenstore/trace.ml                                                                                   0000644 0001750 0001750 00000003010 11146404050 013702  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Tracing for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Trace file descriptor *)
let traceout = ref None

(* Output a trace string *)
let out str =
  match !traceout with
    | Some channel -> Printf.fprintf channel "%s" str; flush channel
    | None -> ()

(* Trace a creation *)
let create data t =
  out (Printf.sprintf "CREATE %s %d\n" t data)

(* Trace a destruction *)
let destroy data t =
  out (Printf.sprintf "DESTROY %s %d\n" t data)

(* Trace I/O *)
let io domain_id prefix time message =
  let message_type = Message.message_type_to_string message.Message.header.Message.message_type
  and sanitised_data = Utils.sanitise_string message.Message.payload in
  out (Printf.sprintf "%s %d %s %s (%s)\n" prefix domain_id time message_type sanitised_data)
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                        xenstore/transaction.ml                                                                             0000644 0001750 0001750 00000032065 11146404050 015145  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Transactions for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type tr =
  {
    domain_id : int;
    transaction_id : int32
  }

type operation =
  | NONE
  | READ
  | WRITE
  | RM

type element =
  {
    transaction : tr;
    operation : operation;
    path : string;
    mutable modified : bool
  }

type changed_domain =
  {
    id : int;
    entries : int
  }

let equal t1 t2 =
  t1.domain_id = t2.domain_id && t1.transaction_id = t2.transaction_id

let fire_watch watches changed_node =
  match changed_node.operation with
  | RM -> watches#fire_watches changed_node.path false true
  | WRITE -> watches#fire_watches changed_node.path false false
  | _ -> ()

let fire_watches watches changed_nodes =
  List.iter (fire_watch watches) changed_nodes

let make domain_id transaction_id =
  {
    domain_id = domain_id;
    transaction_id = transaction_id
  }

let make_element transaction operation path =
  {
    transaction = transaction;
    operation = operation;
    path = path;
    modified = false
  }

module Transaction_hashtbl =
  Hashtbl.Make
  (struct
    type t = tr
    let equal = equal
    let hash = Hashtbl.hash
  end)

class transaction_reads =
object (self)
  val m_paths = Hashtbl.create 32
  val m_transactions = Transaction_hashtbl.create 8
  method private paths = m_paths
  method private transactions = m_transactions
  method add transaction path =
    let operation = make_element transaction READ path
    and paths = self#paths
    and transactions = self#transactions in
    let path_operations =
      if Hashtbl.mem paths path
      then
        let current_operations = Hashtbl.find paths path in
        if not (List.exists (fun op -> transaction = op.transaction) current_operations)
        then operation :: current_operations
        else current_operations
      else [ operation ]
    and transaction_operations =
      if Transaction_hashtbl.mem transactions transaction
      then
        let current_operations = Transaction_hashtbl.find transactions transaction in
        if not (List.exists (fun op -> path = op.path) current_operations)
        then operation :: current_operations
        else current_operations
      else [ operation ] in
    Hashtbl.replace paths path path_operations;
    Transaction_hashtbl.replace transactions transaction transaction_operations
  method path_operations path = Hashtbl.find self#paths path
  method remove_path_operation operation =
    let remaining = List.filter (fun op -> not (equal op.transaction operation.transaction)) (self#path_operations operation.path) in
    if List.length remaining > 0
    then Hashtbl.replace self#paths operation.path remaining
    else Hashtbl.remove self#paths operation.path
  method remove_transaction_operations transaction =
    (try List.iter self#remove_path_operation (self#transaction_operations transaction) with Not_found -> ());
    Transaction_hashtbl.remove self#transactions transaction
  method transaction_operations transaction = Transaction_hashtbl.find self#transactions transaction
end

class ['contents] transaction_store (transaction : tr) (store : 'contents Store.store) (reads : transaction_reads) =
object (self)
  inherit ['contents]Store.store as super
  val m_reads = reads
  val m_store = store
  val m_transaction = transaction
  val m_updates = Hashtbl.create 8
  method private domain_id = self#transaction.domain_id
  method private merge_node node =
    if self#op_exists node#path WRITE || self#op_exists node#path RM || self#op_exists node#path NONE
    then self#store#replace_node node
    else
      match node#contents with
      | Store.Children children | Store.Hack (_, children) -> List.iter (fun child -> self#merge_node child) children
      | _ -> ()
  method private op_add path op =
    match op with
    | WRITE -> if not (self#op_exists path RM) then Hashtbl.replace self#updates path (make_element self#transaction op path)
    | RM -> Hashtbl.replace self#updates path (make_element self#transaction op path)
    | READ -> if not (self#op_exists path READ) then self#reads#add self#transaction path
    | NONE -> Hashtbl.replace self#updates path (make_element self#transaction op path)
  method private op_exists path op =
    match op with
    | WRITE | RM | NONE -> (try (Hashtbl.find self#updates path).operation = op with Not_found -> false)
    | READ -> (try List.exists (fun op -> op.transaction = self#transaction) (self#reads#path_operations path) with Not_found -> false)
  method private reads = m_reads
  method private store = m_store
  method private transaction = m_transaction
  method private updates = m_updates
  method changed_nodes = Hashtbl.fold (fun path element nodes -> element :: nodes) self#updates []
  method create_node path =
    if not (self#op_exists path WRITE) then self#op_add path WRITE;
    super#create_node path
  method merge = self#merge_node self#root
  method node_exists path =
    if self#op_exists path WRITE || self#op_exists path RM || self#op_exists path NONE then super#node_exists path else self#store#node_exists path
  method read_node path =
    if self#op_exists path WRITE || self#op_exists path RM || self#op_exists path NONE
    then super#read_node path
    else (
      self#op_add path READ;
      self#store#read_node path
    )
  method remove_node path =
    let parent_path = Store.parent_path path in
    if self#op_exists parent_path WRITE || self#op_exists parent_path RM || self#op_exists parent_path NONE
    then (
      super#remove_node path;
      self#op_add path RM
    )
    else (
      if not (super#node_exists parent_path)
      then (
        super#create_node parent_path;
        let contents =
          (match (self#store#read_node parent_path) with
            | Store.Children _ -> Store.Children []
            | Store.Hack (value, _) -> Store.Hack (value, [])
            | contents -> contents) in
        (super#get_node parent_path)#set_contents contents
      );
      let self_parent_node = self#get_node parent_path in
      match self_parent_node#contents with
      | Store.Children self_parent_children | Store.Hack (_, self_parent_children) -> (
            (match self#store#read_node parent_path with
              | Store.Children store_parent_children | Store.Hack (_, store_parent_children) -> List.iter (fun store_parent_child -> if not (List.exists (fun self_parent_child -> Store.compare self_parent_child store_parent_child = 0) self_parent_children) then ignore (self_parent_node#add_child store_parent_child)) store_parent_children
              | Store.Empty -> ()
              | Store.Value _ -> raise (Constants.Xs_error (Constants.EINVAL, "Transaction.transaction_store#remove_node", path)));
            self_parent_node#remove_child path;
            self#op_add path RM;
            self#op_add parent_path NONE
          )
      | _ -> raise (Constants.Xs_error (Constants.EINVAL, "Transaction.transaction_store#remove_node", path))
    )
  method write_node path (contents : 'contents) =
    if self#op_exists path WRITE || self#op_exists path RM || self#op_exists path NONE
    then (
      if not (super#node_exists path) then super#create_node path;
      self#op_add path WRITE;
      super#write_node path contents
    )
    else if self#store#node_exists path
    then (
      self#create_node path;
      super#write_node path contents
    )
    else raise (Constants.Xs_error (Constants.EINVAL, "Transaction.transaction_store#write_node", path))
end

class ['contents] transactions (store : 'contents Store.store) =
object (self)
  val m_base_store = store
  val m_num_transactions = Hashtbl.create 8
  val m_reads = new transaction_reads
  val m_transaction_changed_domains = Transaction_hashtbl.create 8
  val m_transaction_ids = Hashtbl.create 8
  val m_transactions = Transaction_hashtbl.create 8
  method private add transaction store =
    if not (Transaction_hashtbl.mem self#transactions transaction)
    then (
      Transaction_hashtbl.add self#transactions transaction (new transaction_store transaction store self#reads);
      Transaction_hashtbl.add self#transaction_changed_domains transaction [ { id = transaction.domain_id; entries = 0 } ];
      Hashtbl.replace self#num_transactions transaction.domain_id (try succ (self#num_transactions_for_domain transaction.domain_id) with Not_found -> 1);
    )
  method private num_transactions = m_num_transactions
  method private reads = m_reads
  method private remove transaction =
    self#reads#remove_transaction_operations transaction;
    Transaction_hashtbl.remove self#transactions transaction;
    Transaction_hashtbl.remove self#transaction_changed_domains transaction;
    Hashtbl.replace self#num_transactions transaction.domain_id (pred (self#num_transactions_for_domain transaction.domain_id))
  method private transaction_changed_domains = m_transaction_changed_domains
  method private transaction_ids = m_transaction_ids
  method private transaction_store transaction = Transaction_hashtbl.find self#transactions transaction
  method private transactions = m_transactions
  method private validate transaction =
    try not (List.fold_left (fun modified op -> if equal op.transaction transaction then op.modified || modified else modified) false (self#reads#transaction_operations transaction))
    with _ -> true
  method base_store = m_base_store
  method commit transaction =
    if self#validate transaction
    then (
      let tstore = self#transaction_store transaction in
      let changed_nodes = tstore#changed_nodes in
      self#invalidate_nodes changed_nodes;
      tstore#merge;
      self#remove transaction;
      changed_nodes
    )
    else (
      self#remove transaction;
      raise Not_found
    )
  method domain_entries transaction = Transaction_hashtbl.find self#transaction_changed_domains transaction
  method domain_entry_decr (transaction : tr) domain_id =
    try
      let domain_entry = List.find (fun entry -> entry.id = domain_id) (self#domain_entries transaction) in
      let new_domain_entry = { id = domain_id; entries = pred domain_entry.entries } in
      Transaction_hashtbl.replace self#transaction_changed_domains transaction (new_domain_entry :: (List.filter (fun entry -> entry.id <> domain_id) (self#domain_entries transaction)))
    with Not_found ->
        let new_domain_entry = { id = domain_id; entries = (- 1) } in
        Transaction_hashtbl.replace self#transaction_changed_domains transaction (new_domain_entry :: (self#domain_entries transaction))
  method domain_entry_incr (transaction : tr) domain_id =
    try
      let domain_entry = List.find (fun entry -> entry.id = domain_id) (self#domain_entries transaction) in
      let new_domain_entry = { id = domain_id; entries = succ domain_entry.entries } in
      Transaction_hashtbl.replace self#transaction_changed_domains transaction (new_domain_entry :: (List.filter (fun entry -> entry.id <> domain_id) (self#domain_entries transaction)))
    with Not_found ->
        let new_domain_entry = { id = domain_id; entries = 1 } in
        Transaction_hashtbl.replace self#transaction_changed_domains transaction (new_domain_entry :: (self#domain_entries transaction))
  method exists transaction = Transaction_hashtbl.mem self#transactions transaction
  method invalidate path = try List.iter (fun op -> op.modified <- true) (self#reads#path_operations path) with Not_found -> ()
  method invalidate_nodes nodes = List.iter (fun node -> self#invalidate node.path) nodes
  method new_transaction (domain : Domain.domain) store =
    if not (Hashtbl.mem self#transaction_ids domain#id) then Hashtbl.add self#transaction_ids domain#id 1l;
    let transaction_id = Hashtbl.find self#transaction_ids domain#id in
    let transaction = make domain#id transaction_id in
    Hashtbl.replace self#transaction_ids domain#id (Int32.succ transaction_id);
    if not (Transaction_hashtbl.mem self#transactions transaction) && transaction.transaction_id <> 0l
    then (self#add transaction store; transaction)
    else self#new_transaction domain store
  method num_transactions_for_domain domain_id = try Hashtbl.find self#num_transactions domain_id with Not_found -> 0
  method remove_domain (domain : Domain.domain) =
    Transaction_hashtbl.iter (fun transaction store -> if transaction.domain_id = domain#id then self#remove transaction) self#transactions;
    Hashtbl.remove self#num_transactions domain#id;
    Hashtbl.remove self#transaction_ids domain#id
  method store transaction = try ((self#transaction_store transaction) :> 'contents Store.store) with Not_found -> self#base_store
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                           xenstore/utils.c                                                                                    0000644 0001750 0001750 00000001727 11146403776 013611  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 #define _GNU_SOURCE
#include <stdio.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <signal.h>
#include "utils.h"

static void default_xprintf(const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vfprintf(stderr, fmt, args);
	va_end(args);
	fflush(stderr);
}

void (*xprintf)(const char *fmt, ...) = default_xprintf;

void barf(const char *fmt, ...)
{
	char *str;
	int bytes;
	va_list arglist;

	xprintf("FATAL: ");

	va_start(arglist, fmt);
	bytes = vasprintf(&str, fmt, arglist);
	va_end(arglist);

 	if (bytes >= 0) {
		xprintf("%s\n", str);
		free(str);
	}
	exit(1);
}

void barf_perror(const char *fmt, ...)
{
	char *str;
	int bytes, err = errno;
	va_list arglist;

	xprintf("FATAL: ");

	va_start(arglist, fmt);
	bytes = vasprintf(&str, fmt, arglist);
	va_end(arglist);

 	if (bytes >= 0) {
		xprintf("%s: %s\n", str, strerror(err));
		free(str);
	}
	exit(1);
}
                                         xenstore/utils.h                                                                                    0000644 0001750 0001750 00000002240 11146403776 013605  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 #ifndef _UTILS_H
#define _UTILS_H
#include <stdbool.h>
#include <string.h>
#include <stdint.h>

/* Is A == B ? */
#define streq(a,b) (strcmp((a),(b)) == 0)

/* Does A start with B ? */
#define strstarts(a,b) (strncmp((a),(b),strlen(b)) == 0)

/* Does A end in B ? */
static inline bool strends(const char *a, const char *b)
{
	if (strlen(a) < strlen(b))
		return false;

	return streq(a + strlen(a) - strlen(b), b);
}

#define ARRAY_SIZE(arr) (sizeof(arr) / sizeof((arr)[0]))

void barf(const char *fmt, ...) __attribute__((noreturn));
void barf_perror(const char *fmt, ...) __attribute__((noreturn));

void (*xprintf)(const char *fmt, ...);

#define eprintf(_fmt, _args...) xprintf("[ERR] %s" _fmt, __FUNCTION__, ##_args)

/*
 * Mux errno values onto returned pointers.
 */

static inline void *ERR_PTR(long error)
{
	return (void *)error;
}

static inline long PTR_ERR(const void *ptr)
{
	return (long)ptr;
}

static inline long IS_ERR(const void *ptr)
{
	return ((unsigned long)ptr > (unsigned long)-1000L);
}


#endif /* _UTILS_H */

/*
 * Local variables:
 *  c-file-style: "linux"
 *  indent-tabs-mode: t
 *  c-indent-level: 8
 *  c-basic-offset: 8
 *  tab-width: 8
 * End:
 */
                                                                                                                                                                                                                                                                                                                                                                xenstore/utils.ml                                                                                   0000644 0001750 0001750 00000007733 11146404050 013764  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Utils for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Print an error to standard output stream and die *)
let barf str =
  Printf.printf "FATAL: %s\n" str; flush stdout;
  ignore (exit 1)

(* Print an error to the error stream and die *)
let barf_perror str =
  Printf.eprintf "FATAL: %s\n" str; flush stderr;
  ignore (exit 1)

(* Convert a string of bytes into an int32 *)
let bytes_to_int32 bytes =
  let num_bytes = 4 in
  (* Convert bytes to an int32 *)
  let rec loop i n =
    if i >= num_bytes
    then n
    else loop (succ i) (Int32.add (Int32.shift_left n 8) (Int32.of_int (int_of_char bytes.[(num_bytes - 1) - i])))
  in
  loop 0 Int32.zero

(* Convert a string of bytes into an int *)
let bytes_to_int bytes =
  Int32.to_int (bytes_to_int32 bytes)

let combine lst =
  List.fold_left (fun rest i -> rest ^ i) Constants.null_string lst

let combine_with_string lst str =
  List.fold_left (fun rest i -> rest ^ i ^ str) Constants.null_string lst

(* Convert an int into a string of bytes *)
let int32_to_bytes num =
  let num_bytes = 4 in
  let bytes = String.create num_bytes in
  let rec loop i n =
    if i < num_bytes
    then (
      bytes.[i] <- char_of_int (Int32.to_int (Int32.logand 0xFFl n));
      loop (succ i) (Int32.shift_right_logical n 8)
    )
  in
  loop 0 num;
  bytes

(* Convert an int into a string of bytes *)
let int_to_bytes num =
  int32_to_bytes (Int32.of_int num)

(* Null terminate a string *)
let null_terminate str =
  str ^ (String.make 1 Constants.null_char)

(* Remove the last element from a list *)
let remove_last list =
  let length = pred (List.length list) in
  let rec loop n = (if (n = length) then [] else (List.nth list n :: loop (succ n))) in
  loop 0

(* Clean a string up for output *)
let sanitise_string str =
  let replacement_string = String.make 1 ' ' in
  let rec replace_nulls s =
    try
      let i = String.index s Constants.null_char in
      (String.sub s 0 i) ^ replacement_string ^ (replace_nulls (String.sub s (succ i) ((String.length s) - (succ i))))
    with Not_found -> s
  in
  replace_nulls str

(* Split a string into a list of strings based on the specified character *)
let split_on_char str char =
  let rec split_loop s =
    if (s = Constants.null_string) then []
    else
      try
        let null_index = String.index s char in
        String.sub s 0 null_index :: split_loop (String.sub s (succ null_index) ((String.length s) - (succ null_index)))
      with Not_found -> [ s ] | Invalid_argument _ -> []
  in
  split_loop str

(* Split a string into a list of strings based on the null character *)
let split str =
  split_on_char str Constants.null_char

(* Strip the trailing null byte off a string, if there is one *)
let strip_null str =
  if String.length str = 0 then str
  else
    let last = pred (String.length str) in
    if str.[last] = Constants.null_char then String.sub str 0 last else str

(* Return if a string contains another string *)
let rec strstr s1 s2 =
  try
    let i = String.index s1 s2.[0] in
    if String.length (String.sub s1 i ((String.length s1) - i)) < String.length s2
    then false
    else if String.sub s1 i (String.length s2) = s2
    then true
    else strstr (String.sub s1 (succ i) ((String.length s1) - (succ i))) s2
  with Not_found -> false
                                     xenstore/watch.ml                                                                                   0000644 0001750 0001750 00000010311 11146404050 013714  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    Watches for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

type t =
  {
    domain : Domain.domain;
    path : string;
    token : string;
    relative : bool
  }

let make domain path token relative =
  {
    domain = domain;
    path = path;
    token = token;
    relative = relative
  }

let equal watch1 watch2 =
  watch1.domain#id = watch2.domain#id && watch1.token = watch2.token && watch1.path = watch2.path

(* Fire a watch *)
let fire_watch path recurse watch =
  let relative_base_path = Store.domain_root ^ (string_of_int watch.domain#id) in
  let relative_base_length = succ (String.length relative_base_path) in
  if Store.is_child path watch.path
  then
    let watch_path =
      if watch.relative
      then String.sub path relative_base_length ((String.length path) - relative_base_length)
      else path in
    watch.domain#add_output_message (Message.event ((Utils.null_terminate watch_path) ^ (Utils.null_terminate watch.token)))
  else if recurse && Store.is_child watch.path path
  then
    let watch_path =
      if watch.relative
      then String.sub watch.path relative_base_length ((String.length watch.path) - relative_base_length)
      else watch.path in
    watch.domain#add_output_message (Message.event ((Utils.null_terminate watch_path) ^ (Utils.null_terminate watch.token)))

class watches =
object(self)
  val m_domain_watches = Hashtbl.create 16
  val m_watches = Hashtbl.create 32
  method private add_domain_watch watch =
    let watches = try Hashtbl.find self#domain_watches watch.domain#id with Not_found -> [] in
    Hashtbl.replace self#domain_watches watch.domain#id (watch :: watches)
  method private domain_watches = m_domain_watches
  method private remove_domain_watch watch =
    let watches = try Hashtbl.find self#domain_watches watch.domain#id with Not_found -> [] in
    Hashtbl.replace self#domain_watches watch.domain#id (List.filter (fun w -> not (equal watch w)) watches)
  method private watches = m_watches
  method add (watch : t) =
    if Hashtbl.mem self#watches watch.path
    then (
      let path_watches = Hashtbl.find self#watches watch.path in
      try ignore (List.find (equal watch) path_watches); false
      with Not_found -> (
            Hashtbl.replace self#watches watch.path (watch :: path_watches);
            self#add_domain_watch watch;
            true
          )
    )
    else (
      Hashtbl.add self#watches watch.path [ watch ];
      self#add_domain_watch watch;
      true
    )
  method fire_watches path in_transaction recursive =
    if not in_transaction then Hashtbl.iter (fun _ watches -> List.iter (fire_watch path recursive) watches) self#watches
  method num_watches_for_domain domain_id = try List.length (Hashtbl.find self#domain_watches domain_id) with Not_found -> 0
  method remove (watch : t) =
    if Hashtbl.mem self#watches watch.path
    then (
      let remaining_watches = List.filter (fun w -> not (equal watch w)) (Hashtbl.find self#watches watch.path) in
      if List.length remaining_watches > 0
      then Hashtbl.replace self#watches watch.path remaining_watches
      else Hashtbl.remove self#watches watch.path;
      self#remove_domain_watch watch;
      true
    )
    else false
  method remove_watches (domain : Domain.domain) =
    if Hashtbl.mem self#domain_watches domain#id
    then (
      List.iter (fun watch -> if self#remove watch then Trace.destroy watch.domain#id "watch") (Hashtbl.find self#domain_watches domain#id);
      Hashtbl.remove self#domain_watches domain#id;
    )
end
                                                                                                                                                                                                                                                                                                                       xenstore/xenbus_c.c                                                                                 0000644 0001750 0001750 00000012057 11146404050 014237  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /*
    XenBus C stubs for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <stdint.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>

#include <xenctrl.h>
#include <xen/io/xs_wire.h>

#include <caml/mlvalues.h>
#include <caml/callback.h>
#include <caml/memory.h>
#include <caml/alloc.h>

/* Memory barrier */
value mb_c (value dummy)
{
	CAMLparam1 (dummy);

    asm volatile ( "lock; addl $0,0(%%esp)" : : : "memory" );

	CAMLreturn (Val_unit);
}

/* Map a file */
value mmap_c (value fd_v)
{
	CAMLparam1 (fd_v);

	int fd = Int_val (fd_v);
	long pagesize = getpagesize();
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) mmap(NULL, pagesize, PROT_READ|PROT_WRITE, MAP_SHARED, fd, 0);

	CAMLreturn (rv);
}

/* Unmap a file */
value munmap_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	long pagesize = getpagesize();

	CAMLreturn (Val_int (munmap(intf, pagesize)));
}

/* Map a foreign page */
value xc_map_foreign_range_c (value xc_handle_v, value domid_v, value mfn_v)
{
	CAMLparam3 (xc_handle_v, domid_v, mfn_v);

	int xc_handle = Int_val (xc_handle_v);
	long pagesize = getpagesize();
	uint32_t domid = (uint32_t)(Int_val (domid_v));
	unsigned long mfn = (unsigned long)(Int_val (mfn_v));
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) xc_map_foreign_range(xc_handle, domid, pagesize, PROT_READ|PROT_WRITE, mfn);

	CAMLreturn (rv);
}

value get_index_c (value index_v)
{
	CAMLparam1 (index_v);

	uint32_t i = *(uint32_t *)(Field (index_v, 0));

	CAMLreturn (caml_copy_int32(i));
}

value set_index_c (value index_v, value val_v)
{
	CAMLparam2 (index_v, val_v);

	uint32_t i = Int32_val (val_v);
	*(uint32_t *)(Field (index_v, 0)) = i;

	CAMLreturn (Val_unit);
}

value init_req_ring_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->req);

	CAMLreturn (rv);
}

value init_rsp_ring_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->rsp);

	CAMLreturn (rv);
}

value init_req_cons_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->req_cons);

	CAMLreturn (rv);
}

value init_req_prod_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->req_prod);

	CAMLreturn (rv);
}

value init_rsp_cons_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv =  alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->rsp_cons);

	CAMLreturn (rv);
}

value init_rsp_prod_c (value xenbus_v)
{
	CAMLparam1 (xenbus_v);

	struct xenstore_domain_interface *intf = (struct xenstore_domain_interface *)Field (xenbus_v, 0);
	value rv = alloc (Abstract_tag, 1);
	Field (rv, 0) = (value) &(intf->rsp_prod);

	CAMLreturn (rv);
}

/* Read from a ring buffer */
value read_ring_c (value ring_v, value ring_ofs_v, value buff_v, value buff_ofs_v, value len_v)
{
	CAMLparam5 (ring_v, ring_ofs_v, buff_v, buff_ofs_v, len_v);

	char *ring = (char *)(Field (ring_v, 0));
	char *buff = String_val (buff_v);
	int ring_ofs = Int_val (ring_ofs_v);
	int buff_ofs = Int_val (buff_ofs_v);
	int len = Int_val (len_v);
	int i;

	for (i = 0; i < len; i++) {
		buff[buff_ofs + i] = ring[ring_ofs + i];
	}

	CAMLreturn (Val_unit);
}

/* Write to a ring buffer */
value write_ring_c (value ring_v, value ring_ofs_v, value buff_v, value buff_ofs_v, value len_v)
{
	CAMLparam5 (ring_v, ring_ofs_v, buff_v, buff_ofs_v, len_v);

	char *ring = (char *)(Field (ring_v, 0));
	char *buff = String_val (buff_v);
	int ring_ofs = Int_val (ring_ofs_v);
	int buff_ofs = Int_val (buff_ofs_v);
	int len = Int_val (len_v);
	int i;

	for (i = 0; i < len; i++) {
		ring[ring_ofs + i] = buff[buff_ofs + i];
	}

	CAMLreturn (Val_unit);
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                 xenstore/xenbus.ml                                                                                  0000644 0001750 0001750 00000011022 11146404050 014112  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    XenBus for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let ring_size = 1024

type xenbus_t
type ring_t
type ring_index_t

external init_req_cons : xenbus_t -> ring_index_t = "init_req_cons_c"
external init_req_prod : xenbus_t -> ring_index_t = "init_req_prod_c"
external init_req_ring : xenbus_t -> ring_t = "init_req_ring_c"
external init_rsp_cons : xenbus_t -> ring_index_t = "init_rsp_cons_c"
external init_rsp_prod : xenbus_t -> ring_index_t = "init_rsp_prod_c"
external init_rsp_ring : xenbus_t -> ring_t = "init_rsp_ring_c"
external read_ring : ring_t -> int -> string -> int -> int -> unit = "read_ring_c"
external write_ring : ring_t -> int -> string -> int -> int -> unit = "write_ring_c"
external get_index : ring_index_t -> int32 = "get_index_c"
external set_index : ring_index_t -> int32 -> unit = "set_index_c"
external mmap : int -> xenbus_t = "mmap_c"
external map_foreign : int -> int -> int -> xenbus_t = "xc_map_foreign_range_c"
external munmap : xenbus_t -> unit = "munmap_c"
external mb : unit -> unit = "mb_c"

(* Ring buffer *)
class ring_buffer ring consumer producer =
object (self)
  val m_consumer = consumer
  val m_producer = producer
  val m_ring = ring
  method private advance_consumer amount = set_index m_consumer (Int32.add self#consumer (Int32.of_int amount))
  method private advance_producer amount = set_index m_producer (Int32.add self#producer (Int32.of_int amount))
  method private check_indexes = self#diff <= ring_size
  method private consumer = get_index m_consumer
  method private diff = Int32.to_int (Int32.sub self#producer self#consumer)
  method private mask_index index = (Int32.to_int index) land (pred ring_size)
  method private producer = get_index m_producer
  method private ring = m_ring
  method private set_producer index = set_index m_producer index
  method can_read = self#diff <> 0
  method can_write = self#diff <> ring_size
  method read buffer offset length =
    let start = self#mask_index self#consumer
    and diff = self#diff in
    if not self#check_indexes then raise (Constants.Xs_error (Constants.EIO, "ring_buffer#read_ring", "could not check indexes"));
    mb ();
    let read_length = min (min diff length) (ring_size - start) in
    read_ring self#ring start buffer offset read_length;
    mb ();
    self#advance_consumer read_length;
    read_length
  method write buffer offset length =
    let start = self#mask_index self#producer
    and diff = self#diff in
    if not self#check_indexes then raise (Constants.Xs_error (Constants.EIO, "ring_buffer#write_ring", "could not check indexes"));
    mb ();
    let write_length = min (min (ring_size - diff) length) (ring_size - start) in
    write_ring self#ring start buffer offset write_length;
    mb ();
    self#advance_producer write_length;
    write_length
end

(* XenBus interface *)
class xenbus_interface port xenbus =
object (self)
  inherit Interface.interface as super
  val m_port = port
  val m_request_ring = new ring_buffer (init_req_ring xenbus) (init_req_cons xenbus) (init_req_prod xenbus)
  val m_response_ring = new ring_buffer (init_rsp_ring xenbus) (init_rsp_cons xenbus) (init_rsp_prod xenbus)
  val m_xenbus = xenbus
  method private port = m_port
  method private request_ring = m_request_ring
  method private response_ring = m_response_ring
  method can_read = self#request_ring#can_read
  method can_write = self#response_ring#can_write
  method destroy = if Eventchan.unbind self#port then munmap m_xenbus
  method read buffer offset length =
    let bytes_read = self#request_ring#read buffer offset (min length (String.length buffer)) in
    Eventchan.notify self#port;
    bytes_read
  method write buffer offset length =
    let bytes_written = self#response_ring#write buffer offset (min length (String.length buffer)) in
    Eventchan.notify self#port;
    bytes_written
end
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                              xenstore/xenstore_client.c                                                                          0000644 0001750 0001750 00000036506 11146403776 015661  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /*
 * This file is subject to the terms and conditions of the GNU General
 * Public License.  See the file "COPYING" in the main directory of
 * this archive for more details.
 *
 * Copyright (C) 2005 by Christian Limpach
 * Copyright (C) 2005 XenSource Ltd.
 *
 */

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <getopt.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#include <xs.h>

#include <sys/ioctl.h>

#define PATH_SEP '/'
#define MAX_PATH_LEN 256

#define MAX_PERMS 16

enum mode {
    MODE_unknown,
    MODE_chmod,
    MODE_exists,
    MODE_list,
    MODE_ls,
    MODE_read,
    MODE_rm,
    MODE_write,
};

static char *output_buf = NULL;
static int output_pos = 0;

static int output_size = 0;

static void
output(const char *fmt, ...) {
    va_list ap;
    int len;
    char buf[1];

    va_start(ap, fmt);
    len = vsnprintf(buf, 1, fmt, ap);
    if (len < 0)
	err(1, "output");
    va_end(ap);
    if (len + 1 + output_pos > output_size) {
	output_size += len + 1024;
	output_buf = realloc(output_buf, output_size);
	if (output_buf == NULL)
	    err(1, "malloc");
    }
    va_start(ap, fmt);
    if (vsnprintf(&output_buf[output_pos], len + 1, fmt, ap) != len)
	err(1, "output");
    va_end(ap);
    output_pos += len;
}

static void
usage(enum mode mode, int incl_mode, const char *progname)
{
    const char *mstr = NULL;

    switch (mode) {
    case MODE_unknown:
	errx(1, "Usage: %s <mode> [-h] [...]", progname);
    case MODE_read:
	mstr = incl_mode ? "read " : "";
	errx(1, "Usage: %s %s[-h] [-p] [-s] key [...]", progname, mstr);
    case MODE_write:
	mstr = incl_mode ? "write " : "";
	errx(1, "Usage: %s %s[-h] [-s] key value [...]", progname, mstr);
    case MODE_rm:
	mstr = incl_mode ? "rm " : "";
	errx(1, "Usage: %s %s[-h] [-s] [-t] key [...]", progname, mstr);
    case MODE_exists:
	mstr = incl_mode ? "exists " : "";
    case MODE_list:
	mstr = mstr ? : incl_mode ? "list " : "";
	errx(1, "Usage: %s %s[-h] [-s] key [...]", progname, mstr);
    case MODE_ls:
	mstr = mstr ? : incl_mode ? "ls " : "";
	errx(1, "Usage: %s %s[-h] [-s] [path]", progname, mstr);
    case MODE_chmod:
	mstr = incl_mode ? "chmod " : "";
	errx(1, "Usage: %s %s[-h] [-s] key <mode [modes...]>", progname, mstr);
    }
}


static int
do_rm(char *path, struct xs_handle *xsh, xs_transaction_t xth)
{
    if (xs_rm(xsh, xth, path)) {
        return 0;
    }
    else {
        warnx("could not remove path %s", path);
        return 1;
    }
}

#define STRING_MAX XENSTORE_ABS_PATH_MAX+1024
static int max_width = 80;
static int desired_width = 60;
static int show_whole_path = 0;

#define TAG " = \"...\""
#define TAG_LEN strlen(TAG)

#define MIN(a, b) (((a) < (b))? (a) : (b))

static void do_ls(struct xs_handle *h, char *path, int cur_depth, int show_perms)
{
    static struct expanding_buffer ebuf;
    char **e;
    char newpath[STRING_MAX], *val;
    int newpath_len;
    int i;
    unsigned int num, len;

    e = xs_directory(h, XBT_NULL, path, &num);
    if (e == NULL)
        err(1, "xs_directory (%s)", path);

    for (i = 0; i<num; i++) {
        char buf[MAX_STRLEN(unsigned int)+1];
        struct xs_permissions *perms;
        unsigned int nperms;
        int linewid;

        /* Compose fullpath */
        newpath_len = snprintf(newpath, sizeof(newpath), "%s%s%s", path, 
                path[strlen(path)-1] == '/' ? "" : "/", 
                e[i]);

        /* Print indent and path basename */
        linewid = 0;
        if (show_whole_path) {
            fputs(newpath, stdout);
        } else {
            for (; linewid<cur_depth; linewid++) {
                putchar(' ');
            }
            linewid += printf("%.*s",
                              (int) (max_width - TAG_LEN - linewid), e[i]);
        }

	/* Fetch value */
        if ( newpath_len < sizeof(newpath) ) {
            val = xs_read(h, XBT_NULL, newpath, &len);
        }
        else {
            /* Path was truncated and thus invalid */
            val = NULL;
            len = 0;
        }

        /* Print value */
        if (val == NULL) {
            printf(":\n");
        }
        else {
            if (max_width < (linewid + len + TAG_LEN)) {
                printf(" = \"%.*s\\...\"",
                       (int)(max_width - TAG_LEN - linewid),
		       sanitise_value(&ebuf, val, len));
            }
            else {
                linewid += printf(" = \"%s\"",
				  sanitise_value(&ebuf, val, len));
                if (show_perms) {
                    putchar(' ');
                    for (linewid++;
                         linewid < MIN(desired_width, max_width);
                         linewid++)
                        putchar((linewid & 1)? '.' : ' ');
                }
            }
        }
        free(val);

        if (show_perms) {
            perms = xs_get_permissions(h, XBT_NULL, newpath, &nperms);
            if (perms == NULL) {
                warn("\ncould not access permissions for %s", e[i]);
            }
            else {
                int i;
                fputs("  (", stdout);
                for (i = 0; i < nperms; i++) {
                    if (i)
                        putchar(',');
                    xs_perm_to_string(perms+i, buf, sizeof(buf));
                    fputs(buf, stdout);
                }
                putchar(')');
            }
        }

        putchar('\n');
            
        do_ls(h, newpath, cur_depth+1, show_perms); 
    }
    free(e);
}

static void
do_chmod(char *path, struct xs_permissions *perms, int nperms, int upto,
	 int recurse, struct xs_handle *xsh, xs_transaction_t xth)
{
    int ret;

    if (!path[0])
	return;

    ret = xs_set_permissions(xsh, xth, path, perms, nperms);
    if (!ret)
	err(1, "Error occurred setting permissions on '%s'", path);

    if (upto) {
	/* apply same permissions to all parent entries: */
	char *path_sep_ptr = strrchr(path, PATH_SEP);
	if (!path_sep_ptr)
	    errx(1, "Unable to locate path separator '%c' in '%s'",
		 PATH_SEP, path);
	
	*path_sep_ptr = '\0'; /* truncate path */
	
	do_chmod(path, perms, nperms, 1, 0, xsh, xth);

	*path_sep_ptr = PATH_SEP;
    }

    if (recurse) {
	char buf[MAX_PATH_LEN];

	/* apply same permissions to all child entries: */
	unsigned int xsval_n;
	char **xsval = xs_directory(xsh, xth, path, &xsval_n);

	if (xsval) {
	    int i;
	    for (i = 0; i < xsval_n; i++) {
		snprintf(buf, MAX_PATH_LEN, "%s/%s", path, xsval[i]);

		do_chmod(buf, perms, nperms, 0, 1, xsh, xth);
	    }

	    free(xsval);
	}
    }
}

static int
perform(enum mode mode, int optind, int argc, char **argv, struct xs_handle *xsh,
        xs_transaction_t xth, int prefix, int tidy, int upto, int recurse)
{
    switch (mode) {
    case MODE_ls:
	if (optind == argc)
	{
	    optind=0;
	    argc=1;
	    argv[0] = "/";
	}
	break;
    default:
	break;
    }

    while (optind < argc) {
        switch (mode) {
        case MODE_unknown:
            /* CANNOT BE REACHED */
            errx(1, "invalid mode %d", mode);
        case MODE_read: {
            static struct expanding_buffer ebuf;
            unsigned len;
            char *val = xs_read(xsh, xth, argv[optind], &len);
            if (val == NULL) {
                warnx("couldn't read path %s", argv[optind]);
                return 1;
            }
            if (prefix)
                output("%s: ", argv[optind]);
            output("%s\n", sanitise_value(&ebuf, val, len));
            free(val);
            optind++;
            break;
        }
        case MODE_write: {
            static struct expanding_buffer ebuf;
            char *val_spec = argv[optind + 1];
            unsigned len;
            expanding_buffer_ensure(&ebuf, strlen(val_spec)+1);
            unsanitise_value(ebuf.buf, &len, val_spec);
            if (!xs_write(xsh, xth, argv[optind], ebuf.buf, len)) {
                warnx("could not write path %s", argv[optind]);
                return 1;
            }
            optind += 2;
        } break;
        case MODE_rm: {
            /* Remove the specified path.  If the tidy flag is set, then also
               remove any containing directories that are both empty and have no
               value attached, and repeat, recursing all the way up to the root if
               necessary.
            */

            char *slash, *path = argv[optind];

            if (tidy) {
                /* Copy path, because we can't modify argv because we will need it
                   again if xs_transaction_end gives us EAGAIN. */
                char *p = malloc(strlen(path) + 1);
                strcpy(p, path);
                path = p;

            again:
                if (do_rm(path, xsh, xth)) {
                    return 1;
                }

                slash = strrchr(p, '/');
                if (slash) {
                    char *val;
                    unsigned len;
                    *slash = '\0';
                    val = xs_read(xsh, xth, p, &len);
                    if (val && len == 0) {
                        unsigned int num;
                        char ** list = xs_directory(xsh, xth, p, &num);

                        if (list && num == 0) {
                            goto again;
                        }
                    }
                }

                free(path);
            }
            else {
                if (do_rm(path, xsh, xth)) {
                    return 1;
                }
            }

            optind++;
            break;
        }
        case MODE_exists: {
            char *val = xs_read(xsh, xth, argv[optind], NULL);
            if (val == NULL) {
                return 1;
            }
            free(val);
            optind++;
            break;
        }
        case MODE_list: {
            unsigned int i, num;
            char **list = xs_directory(xsh, xth, argv[optind], &num);
            if (list == NULL) {
                warnx("could not list path %s", argv[optind]);
                return 1;
            }
            for (i = 0; i < num; i++) {
                if (prefix)
                    output("%s/", argv[optind]);
                output("%s\n", list[i]);
            }
            free(list);
	    optind++;
	    break;
	}
	case MODE_ls: {
	    do_ls(xsh, argv[optind], 0, prefix);
 	    optind++;
 	    break;
        }
        case MODE_chmod: {
            struct xs_permissions perms[MAX_PERMS];
            int nperms = 0;
            /* save path pointer: */
            char *path = argv[optind++];
            for (; argv[optind]; optind++, nperms++)
            {
                if (MAX_PERMS <= nperms)
                    errx(1, "Too many permissions specified.  "
			 "Maximum per invocation is %d.", MAX_PERMS);

                perms[nperms].id = atoi(argv[optind]+1);

                switch (argv[optind][0])
                {
                case 'n':
                    perms[nperms].perms = XS_PERM_NONE;
                    break;
                case 'r':
                    perms[nperms].perms = XS_PERM_READ;
                    break;
                case 'w':
                    perms[nperms].perms = XS_PERM_WRITE;
                    break;
                case 'b':
                    perms[nperms].perms = XS_PERM_READ | XS_PERM_WRITE;
                    break;
                default:
                    errx(1, "Invalid permission specification: '%c'",
			 argv[optind][0]);
                }
            }

            do_chmod(path, perms, nperms, upto, recurse, xsh, xth);
            break;
        }
        }
    }

    return 0;
}

static enum mode lookup_mode(const char *m)
{
    if (strcmp(m, "read") == 0)
	return MODE_read;
    else if (strcmp(m, "chmod") == 0)
	return MODE_chmod;
    else if (strcmp(m, "exists") == 0)
	return MODE_exists;
    else if (strcmp(m, "list") == 0)
	return MODE_list;
    else if (strcmp(m, "ls") == 0)
	return MODE_ls;
    else if (strcmp(m, "rm") == 0)
	return MODE_rm;
    else if (strcmp(m, "write") == 0)
	return MODE_write;
    else if (strcmp(m, "read") == 0)
	return MODE_read;

    errx(1, "unknown mode %s\n", m);
    return 0;
}

int
main(int argc, char **argv)
{
    struct xs_handle *xsh;
    xs_transaction_t xth = XBT_NULL;
    int ret = 0, socket = 0;
    int prefix = 0;
    int tidy = 0;
    int upto = 0;
    int recurse = 0;
    int transaction;
    struct winsize ws;
    enum mode mode;

    const char *_command = strrchr(argv[0], '/');
    const char *command = _command ? &_command[1] : argv[0];
    int switch_argv = -1; /* which element of argv did we switch on */

    if (strncmp(command, "xenstore-", strlen("xenstore-")) == 0)
    {
	switch_argv = 0;
	command = command + strlen("xenstore-");
    }
    else if (argc < 2)
	usage(MODE_unknown, 0, argv[0]);
    else
    {
	command = argv[1];
	switch_argv = 1;
    }

    mode = lookup_mode(command);

    while (1) {
	int c, index = 0;
	static struct option long_options[] = {
	    {"help",    0, 0, 'h'},
	    {"socket",  0, 0, 's'},
	    {"prefix",  0, 0, 'p'}, /* MODE_read || MODE_list */
	    {"tidy",    0, 0, 't'}, /* MODE_rm */
	    {"upto",    0, 0, 'u'}, /* MODE_chmod */
	    {"recurse", 0, 0, 'r'}, /* MODE_chmod */
	    {0, 0, 0, 0}
	};

	c = getopt_long(argc - switch_argv, argv + switch_argv, "fhsptur",
			long_options, &index);
	if (c == -1)
	    break;

	switch (c) {
	case 'h':
	    usage(mode, switch_argv, argv[0]);
	    /* NOTREACHED */
        case 'f':
	    if ( mode == MODE_read || mode == MODE_list || mode == MODE_ls ) {
		max_width = INT_MAX/2;
		desired_width = 0;
		show_whole_path = 1;
	    } else {
		usage(mode, switch_argv, argv[0]);
	    }
            break;
        case 's':
            socket = 1;
            break;
	case 'p':
	    if ( mode == MODE_read || mode == MODE_list || mode == MODE_ls )
		prefix = 1;
	    else
		usage(mode, switch_argv, argv[0]);
	    break;
	case 't':
	    if ( mode == MODE_rm )
		tidy = 1;
	    else
		usage(mode, switch_argv, argv[0]);
	    break;
	case 'u':
	    if ( mode == MODE_chmod )
		upto = 1;
	    else
		usage(mode, switch_argv, argv[0]);
	    break;
	case 'r':
	    if ( mode == MODE_chmod )
		recurse = 1;
	    else
		usage(mode, switch_argv, argv[0]);
	    break;
	}
    }

    switch (mode) {
    case MODE_ls:
	break;
    case MODE_write:
	if ((argc - switch_argv - optind) % 2 == 1) {
	    usage(mode, switch_argv, argv[0]);
	    /* NOTREACHED */
	}
	/* DROP-THRU */
    default:
	if (optind == argc - switch_argv) {
	    usage(mode, switch_argv, argv[0]);
	    /* NOTREACHED */
	}
    }

    switch (mode) {
    case MODE_read:
	transaction = (argc - switch_argv - optind) > 1;
	break;
    case MODE_write:
	transaction = (argc - switch_argv - optind) > 2;
	break;
    case MODE_ls:
	transaction = 0;
	break;
    default:
	transaction = 1;
	break;
    }

    if ( mode == MODE_ls )
    {
	memset(&ws, 0, sizeof(ws));
	ret = ioctl(STDOUT_FILENO, TIOCGWINSZ, &ws);
	if (!ret)
	    max_width = ws.ws_col - 2;
    }

    xsh = socket ? xs_daemon_open() : xs_domain_open();
    if (xsh == NULL)
	err(1, socket ? "xs_daemon_open" : "xs_domain_open");

again:
    if (transaction) {
	xth = xs_transaction_start(xsh);
	if (xth == XBT_NULL)
	    errx(1, "couldn't start transaction");
    }

    ret = perform(mode, optind, argc - switch_argv, argv + switch_argv, xsh, xth, prefix, tidy, upto, recurse);

    if (transaction && !xs_transaction_end(xsh, xth, ret)) {
	if (ret == 0 && errno == EAGAIN) {
	    output_pos = 0;
	    goto again;
	}
	errx(1, "couldn't end transaction");
    }

    if (output_pos)
	printf("%s", output_buf);

    return ret;
}
                                                                                                                                                                                          xenstore/xenstore_control.c                                                                         0000644 0001750 0001750 00000001031 11146403776 016044  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 #include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "xs.h"


int main(int argc, char **argv)
{
  struct xs_handle * xsh;

  if (argc < 2 ||
      strcmp(argv[1], "check"))
  {
    fprintf(stderr,
            "Usage:\n"
            "\n"
            "       %s check\n"
            "\n", argv[0]);
    return 2;
  }

  xsh = xs_daemon_open();

  if (xsh == NULL) {
    fprintf(stderr, "Failed to contact Xenstored.\n");
    return 1;
  }

  xs_debug_command(xsh, argv[1], NULL, 0);

  xs_daemon_close(xsh);

  return 0;
}
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                       xenstore/xenstored.ml                                                                               0000644 0001750 0001750 00000014375 11146404050 014637  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 (* 
    OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let domxs_init id =
  let port = Eventchan.bind_interdomain id (Os.get_xenbus_port ()) in
  let interface = Os.map_xenbus port in
  let connection = new Connection.connection interface in
  Eventchan.notify port;
  new Domain.domain id connection

let domain_entry_change domains domain_entry =
  if (domain_entry.Transaction.entries > 0)
  then
    for i = 1 to domain_entry.Transaction.entries do
      domains#entry_incr domain_entry.Transaction.id
    done
  else if (domain_entry.Transaction.entries < 0)
  then
    for i = domain_entry.Transaction.entries to (- 1) do
      domains#entry_decr domain_entry.Transaction.id
    done

class xenstored options store =
object(self)
  val m_domains = new Domain.domains
  val m_options : Option.t = options
  val m_permissions = new Permission.permissions
  val m_transactions = new Transaction.transactions store
  val m_store = store
  val m_watches = new Watch.watches
  val mutable m_virq_port = Constants.null_file_descr
  initializer m_permissions#set [ (Permission.string_of_permission (Permission.make Permission.NONE 0)) ] store Store.root_path; self#initialise_store
  method private store = m_store
  method add_domain domain =
    self#domains#add domain;
    Trace.create domain#id "connection"
  method add_watch (domain : Domain.domain) watch =
    if not (Domain.is_unprivileged domain) || self#watches#num_watches_for_domain domain#id < self#options.Option.quota_num_watches_per_domain
    then self#watches#add watch
    else raise (Constants.Xs_error (Constants.E2BIG, "Xenstored.xenstored#add_watch", "Too many watches"))
  method commit transaction =
    try
      List.iter (domain_entry_change self#domains) (self#transactions#domain_entries transaction);
      Transaction.fire_watches self#watches (self#transactions#commit transaction);
      true
    with _ -> false
  method domain_entry_count transaction (domain_id : int) =
    let entries = try self#domains#entry_count transaction.Transaction.domain_id with Not_found -> 0 in
    try
      let transaction_entries = (List.find (fun entry -> entry.Transaction.id = transaction.Transaction.domain_id) (self#transactions#domain_entries transaction)).Transaction.entries in
      transaction_entries + entries
    with Not_found -> entries
  method domain_entry_decr store transaction path =
    let domain_id = (List.hd (self#permissions#get store path)).Permission.domain_id in
    if Domain.is_unprivileged_id domain_id then
      if transaction.Transaction.transaction_id <> 0l
      then self#transactions#domain_entry_decr transaction domain_id
      else self#domains#entry_decr domain_id
  method domain_entry_incr store transaction path =
    let domain_id = (List.hd (self#permissions#get store path)).Permission.domain_id in
    if Domain.is_unprivileged_id domain_id then
      if transaction.Transaction.transaction_id <> 0l
      then (
        self#transactions#domain_entry_incr transaction domain_id;
        let entry_count = (List.find (fun entry -> entry.Transaction.id = domain_id) (self#transactions#domain_entries transaction)).Transaction.entries in
        let entry_count_current = try self#domains#entry_count domain_id with Not_found -> 0 in
        if entry_count + entry_count_current > self#options.Option.quota_num_entries_per_domain
        then (
          self#transactions#domain_entry_decr transaction domain_id;
          raise (Constants.Xs_error (Constants.EINVAL, "Xenstored.xenstored#domain_entry_incr", path))
        )
      )
      else (
        self#domains#entry_incr domain_id;
        let entry_count = self#domains#entry_count domain_id in
        if entry_count > self#options.Option.quota_num_entries_per_domain
        then (
          self#domains#entry_decr domain_id;
          raise (Constants.Xs_error (Constants.EINVAL, "Xenstored.xenstored#domain_entry_incr", path))
        )
      )
  method domains = m_domains
  method initialise_domains =
    if self#options.Option.domain_init
    then (
      if Domain.xc_handle = Constants.null_file_descr then Utils.barf_perror "Failed to open connection to hypervisor\n";
      Eventchan.init ();
      let dom0 =
        if self#options.Option.separate_domain
        then (
          self#add_domain (domxs_init (Os.get_domxs_id ()));
          Domain.domu_init 0 (Os.get_dom0_port ()) (Os.get_dom0_mfn ()) true
        )
        else domxs_init 0 in
      m_virq_port <- Eventchan.bind_virq Constants.virq_dom_exc;
      if m_virq_port = Constants.null_file_descr then Utils.barf_perror "Failed to bind to domain exception virq port\n";
      self#add_domain dom0;
      Eventchan.get_channel ()
    )
    else Constants.null_file_descr
  method initialise_store =
    let path = Store.root_path ^ "tool" ^ Store.dividor_str ^ "xenstored" in
    self#store#create_node path;
    self#permissions#add self#store path 0
  method new_transaction domain store =
    if not (Domain.is_unprivileged domain) || self#transactions#num_transactions_for_domain domain#id < self#options.Option.quota_max_transaction
    then self#transactions#new_transaction domain store
    else raise (Constants.Xs_error (Constants.ENOSPC, "Xenstored.xenstored#new_transaction", "Too many transactions"))
  method options = m_options
  method permissions = m_permissions
  method remove_domain domain =
    self#domains#remove domain;
    Trace.destroy domain#id "connection";
    self#watches#remove_watches domain;
    self#transactions#remove_domain domain
  method transactions = m_transactions
  method virq_port = m_virq_port
  method watches = m_watches
end
                                                                                                                                                                                                                                                                   xenstore/xs.c                                                                                       0000644 0001750 0001750 00000050574 11146403776 013107  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /* 
    Xen Store Daemon interface providing simple tree-like database.
    Copyright (C) 2005 Rusty Russell IBM Corporation

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/uio.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <string.h>
#include <unistd.h>
#include <stdbool.h>
#include <stdlib.h>
#include <assert.h>
#include <stdio.h>
#include <signal.h>
#include <stdint.h>
#include <errno.h>
#include "xs.h"
#include "list.h"
#include "utils.h"

struct xs_stored_msg {
	struct list_head list;
	struct xsd_sockmsg hdr;
	char *body;
};

#ifdef USE_PTHREAD

#include <pthread.h>

struct xs_handle {
	/* Communications channel to xenstore daemon. */
	int fd;

	/*
         * A read thread which pulls messages off the comms channel and
         * signals waiters.
         */
	pthread_t read_thr;
	int read_thr_exists;

	/*
         * A list of fired watch messages, protected by a mutex. Users can
         * wait on the conditional variable until a watch is pending.
         */
	struct list_head watch_list;
	pthread_mutex_t watch_mutex;
	pthread_cond_t watch_condvar;

	/* Clients can select() on this pipe to wait for a watch to fire. */
	int watch_pipe[2];

	/*
         * A list of replies. Currently only one will ever be outstanding
         * because we serialise requests. The requester can wait on the
         * conditional variable for its response.
         */
	struct list_head reply_list;
	pthread_mutex_t reply_mutex;
	pthread_cond_t reply_condvar;

	/* One request at a time. */
	pthread_mutex_t request_mutex;
};

#define mutex_lock(m)		pthread_mutex_lock(m)
#define mutex_unlock(m)		pthread_mutex_unlock(m)
#define condvar_signal(c)	pthread_cond_signal(c)
#define condvar_wait(c,m,hnd)	pthread_cond_wait(c,m)

static void *read_thread(void *arg);

#else /* !defined(USE_PTHREAD) */

struct xs_handle {
	int fd;
	struct list_head reply_list;
	struct list_head watch_list;
	/* Clients can select() on this pipe to wait for a watch to fire. */
	int watch_pipe[2];
};

#define mutex_lock(m)		((void)0)
#define mutex_unlock(m)		((void)0)
#define condvar_signal(c)	((void)0)
#define condvar_wait(c,m,hnd)	read_message(hnd)

#endif

static int read_message(struct xs_handle *h);

int xs_fileno(struct xs_handle *h)
{
	char c = 0;

	mutex_lock(&h->watch_mutex);

	if ((h->watch_pipe[0] == -1) && (pipe(h->watch_pipe) != -1)) {
		/* Kick things off if the watch list is already non-empty. */
		if (!list_empty(&h->watch_list))
			while (write(h->watch_pipe[1], &c, 1) != 1)
				continue;
	}

	mutex_unlock(&h->watch_mutex);

	return h->watch_pipe[0];
}

static int get_socket(const char *connect_to)
{
	struct sockaddr_un addr;
	int sock, saved_errno, flags;

	sock = socket(PF_UNIX, SOCK_STREAM, 0);
	if (sock < 0)
		return -1;

	if ((flags = fcntl(sock, F_GETFD)) < 0)
		goto error;
	flags |= FD_CLOEXEC;
	if (fcntl(sock, F_SETFD, flags) < 0)
		goto error;

	addr.sun_family = AF_UNIX;
	strcpy(addr.sun_path, connect_to);

	if (connect(sock, (struct sockaddr *)&addr, sizeof(addr)) != 0)
		goto error;

	return sock;

error:
	saved_errno = errno;
	close(sock);
	errno = saved_errno;
	return -1;
}

static int get_dev(const char *connect_to)
{
	return open(connect_to, O_RDWR);
}

static struct xs_handle *get_handle(const char *connect_to)
{
	struct stat buf;
	struct xs_handle *h = NULL;
	int fd = -1, saved_errno;

	if (stat(connect_to, &buf) != 0)
		return NULL;

	if (S_ISSOCK(buf.st_mode))
		fd = get_socket(connect_to);
	else
		fd = get_dev(connect_to);

	if (fd == -1)
		return NULL;

	h = malloc(sizeof(*h));
	if (h == NULL) {
		saved_errno = errno;
		close(fd);
		errno = saved_errno;
		return NULL;
	}

	memset(h, 0, sizeof(*h));

	h->fd = fd;

	INIT_LIST_HEAD(&h->reply_list);
	INIT_LIST_HEAD(&h->watch_list);

	/* Watch pipe is allocated on demand in xs_fileno(). */
	h->watch_pipe[0] = h->watch_pipe[1] = -1;

#ifdef USE_PTHREAD
	pthread_mutex_init(&h->watch_mutex, NULL);
	pthread_cond_init(&h->watch_condvar, NULL);

	pthread_mutex_init(&h->reply_mutex, NULL);
	pthread_cond_init(&h->reply_condvar, NULL);

	pthread_mutex_init(&h->request_mutex, NULL);
#endif

	return h;
}

struct xs_handle *xs_daemon_open(void)
{
	return get_handle(xs_daemon_socket());
}

struct xs_handle *xs_daemon_open_readonly(void)
{
	return get_handle(xs_daemon_socket_ro());
}

struct xs_handle *xs_domain_open(void)
{
	return get_handle(xs_domain_dev());
}

void xs_daemon_close(struct xs_handle *h)
{
	struct xs_stored_msg *msg, *tmsg;

	mutex_lock(&h->request_mutex);
	mutex_lock(&h->reply_mutex);
	mutex_lock(&h->watch_mutex);

#ifdef USE_PTHREAD
	if (h->read_thr_exists) {
		/* XXX FIXME: May leak an unpublished message buffer. */
		pthread_cancel(h->read_thr);
		pthread_join(h->read_thr, NULL);
	}
#endif

	list_for_each_entry_safe(msg, tmsg, &h->reply_list, list) {
		free(msg->body);
		free(msg);
	}

	list_for_each_entry_safe(msg, tmsg, &h->watch_list, list) {
		free(msg->body);
		free(msg);
	}

	mutex_unlock(&h->request_mutex);
	mutex_unlock(&h->reply_mutex);
	mutex_unlock(&h->watch_mutex);

	if (h->watch_pipe[0] != -1) {
		close(h->watch_pipe[0]);
		close(h->watch_pipe[1]);
	}

	close(h->fd);

	free(h);
}

static bool read_all(int fd, void *data, unsigned int len)
{
	while (len) {
		int done;

		done = read(fd, data, len);
		if (done < 0) {
			if (errno == EINTR)
				continue;
			return false;
		}
		if (done == 0) {
			/* It closed fd on us?  EBADF is appropriate. */
			errno = EBADF;
			return false;
		}
		data += done;
		len -= done;
	}

	return true;
}

#ifdef XSTEST
#define read_all read_all_choice
#define xs_write_all write_all_choice
#endif

static int get_error(const char *errorstring)
{
	unsigned int i;

	for (i = 0; !streq(errorstring, xsd_errors[i].errstring); i++)
		if (i == ARRAY_SIZE(xsd_errors) - 1)
			return EINVAL;
	return xsd_errors[i].errnum;
}

/* Adds extra nul terminator, because we generally (always?) hold strings. */
static void *read_reply(
	struct xs_handle *h, enum xsd_sockmsg_type *type, unsigned int *len)
{
	struct xs_stored_msg *msg;
	char *body;

#ifdef USE_PTHREAD
	/* Read from comms channel ourselves if there is no reader thread. */
	if (!h->read_thr_exists && (read_message(h) == -1))
		return NULL;
#endif

	mutex_lock(&h->reply_mutex);
	while (list_empty(&h->reply_list))
		condvar_wait(&h->reply_condvar, &h->reply_mutex, h);
	msg = list_top(&h->reply_list, struct xs_stored_msg, list);
	list_del(&msg->list);
	assert(list_empty(&h->reply_list));
	mutex_unlock(&h->reply_mutex);

	*type = msg->hdr.type;
	if (len)
		*len = msg->hdr.len;
	body = msg->body;

	free(msg);

	return body;
}

/* Send message to xs, get malloc'ed reply.  NULL and set errno on error. */
static void *xs_talkv(struct xs_handle *h, xs_transaction_t t,
		      enum xsd_sockmsg_type type,
		      const struct iovec *iovec,
		      unsigned int num_vecs,
		      unsigned int *len)
{
	struct xsd_sockmsg msg;
	void *ret = NULL;
	int saved_errno;
	unsigned int i;
	struct sigaction ignorepipe, oldact;

	msg.tx_id = t;
	msg.req_id = 0;
	msg.type = type;
	msg.len = 0;
	for (i = 0; i < num_vecs; i++)
		msg.len += iovec[i].iov_len;

	if (msg.len > XENSTORE_PAYLOAD_MAX) {
		errno = E2BIG;
		return 0;
	}

	ignorepipe.sa_handler = SIG_IGN;
	sigemptyset(&ignorepipe.sa_mask);
	ignorepipe.sa_flags = 0;
	sigaction(SIGPIPE, &ignorepipe, &oldact);

	mutex_lock(&h->request_mutex);

	if (!xs_write_all(h->fd, &msg, sizeof(msg)))
		goto fail;

	for (i = 0; i < num_vecs; i++)
		if (!xs_write_all(h->fd, iovec[i].iov_base, iovec[i].iov_len))
			goto fail;

	ret = read_reply(h, &msg.type, len);
	if (!ret)
		goto fail;

	mutex_unlock(&h->request_mutex);

	sigaction(SIGPIPE, &oldact, NULL);
	if (msg.type == XS_ERROR) {
		saved_errno = get_error(ret);
		free(ret);
		errno = saved_errno;
		return NULL;
	}

	if (msg.type != type) {
		free(ret);
		saved_errno = EBADF;
		goto close_fd;
	}
	return ret;

fail:
	/* We're in a bad state, so close fd. */
	saved_errno = errno;
	mutex_unlock(&h->request_mutex);
	sigaction(SIGPIPE, &oldact, NULL);
close_fd:
	close(h->fd);
	h->fd = -1;
	errno = saved_errno;
	return NULL;
}

/* free(), but don't change errno. */
static void free_no_errno(void *p)
{
	int saved_errno = errno;
	free(p);
	errno = saved_errno;
}

/* Simplified version of xs_talkv: single message. */
static void *xs_single(struct xs_handle *h, xs_transaction_t t,
		       enum xsd_sockmsg_type type,
		       const char *string,
		       unsigned int *len)
{
	struct iovec iovec;

	iovec.iov_base = (void *)string;
	iovec.iov_len = strlen(string) + 1;
	return xs_talkv(h, t, type, &iovec, 1, len);
}

static bool xs_bool(char *reply)
{
	if (!reply)
		return false;
	free(reply);
	return true;
}

char **xs_directory(struct xs_handle *h, xs_transaction_t t,
		    const char *path, unsigned int *num)
{
	char *strings, *p, **ret;
	unsigned int len;

	strings = xs_single(h, t, XS_DIRECTORY, path, &len);
	if (!strings)
		return NULL;

	/* Count the strings. */
	*num = xs_count_strings(strings, len);

	/* Transfer to one big alloc for easy freeing. */
	ret = malloc(*num * sizeof(char *) + len);
	if (!ret) {
		free_no_errno(strings);
		return NULL;
	}
	memcpy(&ret[*num], strings, len);
	free_no_errno(strings);

	strings = (char *)&ret[*num];
	for (p = strings, *num = 0; p < strings + len; p += strlen(p) + 1)
		ret[(*num)++] = p;
	return ret;
}

/* Get the value of a single file, nul terminated.
 * Returns a malloced value: call free() on it after use.
 * len indicates length in bytes, not including the nul.
 */
void *xs_read(struct xs_handle *h, xs_transaction_t t,
	      const char *path, unsigned int *len)
{
	return xs_single(h, t, XS_READ, path, len);
}

/* Write the value of a single file.
 * Returns false on failure.
 */
bool xs_write(struct xs_handle *h, xs_transaction_t t,
	      const char *path, const void *data, unsigned int len)
{
	struct iovec iovec[2];

	iovec[0].iov_base = (void *)path;
	iovec[0].iov_len = strlen(path) + 1;
	iovec[1].iov_base = (void *)data;
	iovec[1].iov_len = len;

	return xs_bool(xs_talkv(h, t, XS_WRITE, iovec,
				ARRAY_SIZE(iovec), NULL));
}

/* Create a new directory.
 * Returns false on failure, or success if it already exists.
 */
bool xs_mkdir(struct xs_handle *h, xs_transaction_t t,
	      const char *path)
{
	return xs_bool(xs_single(h, t, XS_MKDIR, path, NULL));
}

/* Destroy a file or directory (directories must be empty).
 * Returns false on failure, or success if it doesn't exist.
 */
bool xs_rm(struct xs_handle *h, xs_transaction_t t,
	   const char *path)
{
	return xs_bool(xs_single(h, t, XS_RM, path, NULL));
}

/* Get permissions of node (first element is owner).
 * Returns malloced array, or NULL: call free() after use.
 */
struct xs_permissions *xs_get_permissions(struct xs_handle *h,
					  xs_transaction_t t,
					  const char *path, unsigned int *num)
{
	char *strings;
	unsigned int len;
	struct xs_permissions *ret;

	strings = xs_single(h, t, XS_GET_PERMS, path, &len);
	if (!strings)
		return NULL;

	/* Count the strings: each one perms then domid. */
	*num = xs_count_strings(strings, len);

	/* Transfer to one big alloc for easy freeing. */
	ret = malloc(*num * sizeof(struct xs_permissions));
	if (!ret) {
		free_no_errno(strings);
		return NULL;
	}

	if (!xs_strings_to_perms(ret, *num, strings)) {
		free_no_errno(ret);
		ret = NULL;
	}

	free(strings);
	return ret;
}

/* Set permissions of node (must be owner).
 * Returns false on failure.
 */
bool xs_set_permissions(struct xs_handle *h,
			xs_transaction_t t,
			const char *path,
			struct xs_permissions *perms,
			unsigned int num_perms)
{
	unsigned int i;
	struct iovec iov[1+num_perms];

	iov[0].iov_base = (void *)path;
	iov[0].iov_len = strlen(path) + 1;
	
	for (i = 0; i < num_perms; i++) {
		char buffer[MAX_STRLEN(unsigned int)+1];

		if (!xs_perm_to_string(&perms[i], buffer, sizeof(buffer)))
			goto unwind;

		iov[i+1].iov_base = strdup(buffer);
		iov[i+1].iov_len = strlen(buffer) + 1;
		if (!iov[i+1].iov_base)
			goto unwind;
	}

	if (!xs_bool(xs_talkv(h, t, XS_SET_PERMS, iov, 1+num_perms, NULL)))
		goto unwind;
	for (i = 0; i < num_perms; i++)
		free(iov[i+1].iov_base);
	return true;

unwind:
	num_perms = i;
	for (i = 0; i < num_perms; i++)
		free_no_errno(iov[i+1].iov_base);
	return false;
}

/* Watch a node for changes (poll on fd to detect, or call read_watch()).
 * When the node (or any child) changes, fd will become readable.
 * Token is returned when watch is read, to allow matching.
 * Returns false on failure.
 */
bool xs_watch(struct xs_handle *h, const char *path, const char *token)
{
	struct iovec iov[2];

#ifdef USE_PTHREAD
	/* We dynamically create a reader thread on demand. */
	mutex_lock(&h->request_mutex);
	if (!h->read_thr_exists) {
		if (pthread_create(&h->read_thr, NULL, read_thread, h) != 0) {
			mutex_unlock(&h->request_mutex);
			return false;
		}
		h->read_thr_exists = 1;
	}
	mutex_unlock(&h->request_mutex);
#endif

	iov[0].iov_base = (void *)path;
	iov[0].iov_len = strlen(path) + 1;
	iov[1].iov_base = (void *)token;
	iov[1].iov_len = strlen(token) + 1;

	return xs_bool(xs_talkv(h, XBT_NULL, XS_WATCH, iov,
				ARRAY_SIZE(iov), NULL));
}

/* Find out what node change was on (will block if nothing pending).
 * Returns array of two pointers: path and token, or NULL.
 * Call free() after use.
 */
char **xs_read_watch(struct xs_handle *h, unsigned int *num)
{
	struct xs_stored_msg *msg;
	char **ret, *strings, c = 0;
	unsigned int num_strings, i;

	mutex_lock(&h->watch_mutex);

	/* Wait on the condition variable for a watch to fire. */
	while (list_empty(&h->watch_list))
		condvar_wait(&h->watch_condvar, &h->watch_mutex, h);
	msg = list_top(&h->watch_list, struct xs_stored_msg, list);
	list_del(&msg->list);

	/* Clear the pipe token if there are no more pending watches. */
	if (list_empty(&h->watch_list) && (h->watch_pipe[0] != -1))
		while (read(h->watch_pipe[0], &c, 1) != 1)
			continue;

	mutex_unlock(&h->watch_mutex);

	assert(msg->hdr.type == XS_WATCH_EVENT);

	strings     = msg->body;
	num_strings = xs_count_strings(strings, msg->hdr.len);

	ret = malloc(sizeof(char*) * num_strings + msg->hdr.len);
	if (!ret) {
		free_no_errno(strings);
		free_no_errno(msg);
		return NULL;
	}

	ret[0] = (char *)(ret + num_strings);
	memcpy(ret[0], strings, msg->hdr.len);

	free(strings);
	free(msg);

	for (i = 1; i < num_strings; i++)
		ret[i] = ret[i - 1] + strlen(ret[i - 1]) + 1;

	*num = num_strings;

	return ret;
}

/* Remove a watch on a node.
 * Returns false on failure (no watch on that node).
 */
bool xs_unwatch(struct xs_handle *h, const char *path, const char *token)
{
	struct iovec iov[2];

	iov[0].iov_base = (char *)path;
	iov[0].iov_len = strlen(path) + 1;
	iov[1].iov_base = (char *)token;
	iov[1].iov_len = strlen(token) + 1;

	return xs_bool(xs_talkv(h, XBT_NULL, XS_UNWATCH, iov,
				ARRAY_SIZE(iov), NULL));
}

/* Start a transaction: changes by others will not be seen during this
 * transaction, and changes will not be visible to others until end.
 * Returns XBT_NULL on failure.
 */
xs_transaction_t xs_transaction_start(struct xs_handle *h)
{
	char *id_str;
	xs_transaction_t id;

	id_str = xs_single(h, XBT_NULL, XS_TRANSACTION_START, "", NULL);
	if (id_str == NULL)
		return XBT_NULL;

	id = strtoul(id_str, NULL, 0);
	free(id_str);

	return id;
}

/* End a transaction.
 * If abandon is true, transaction is discarded instead of committed.
 * Returns false on failure, which indicates an error: transactions will
 * not fail spuriously.
 */
bool xs_transaction_end(struct xs_handle *h, xs_transaction_t t,
			bool abort)
{
	char abortstr[2];

	if (abort)
		strcpy(abortstr, "F");
	else
		strcpy(abortstr, "T");
	
	return xs_bool(xs_single(h, t, XS_TRANSACTION_END, abortstr, NULL));
}

/* Introduce a new domain.
 * This tells the store daemon about a shared memory page and event channel
 * associated with a domain: the domain uses these to communicate.
 */
bool xs_introduce_domain(struct xs_handle *h,
			 unsigned int domid, unsigned long mfn,
			 unsigned int eventchn)
{
	char domid_str[MAX_STRLEN(domid)];
	char mfn_str[MAX_STRLEN(mfn)];
	char eventchn_str[MAX_STRLEN(eventchn)];
	struct iovec iov[3];

	snprintf(domid_str, sizeof(domid_str), "%u", domid);
	snprintf(mfn_str, sizeof(mfn_str), "%lu", mfn);
	snprintf(eventchn_str, sizeof(eventchn_str), "%u", eventchn);

	iov[0].iov_base = domid_str;
	iov[0].iov_len = strlen(domid_str) + 1;
	iov[1].iov_base = mfn_str;
	iov[1].iov_len = strlen(mfn_str) + 1;
	iov[2].iov_base = eventchn_str;
	iov[2].iov_len = strlen(eventchn_str) + 1;

	return xs_bool(xs_talkv(h, XBT_NULL, XS_INTRODUCE, iov,
				ARRAY_SIZE(iov), NULL));
}

bool xs_set_target(struct xs_handle *h,
			 unsigned int domid, unsigned int target)
{
	char domid_str[MAX_STRLEN(domid)];
	char target_str[MAX_STRLEN(target)];
	struct iovec iov[2];

	snprintf(domid_str, sizeof(domid_str), "%u", domid);
	snprintf(target_str, sizeof(target_str), "%u", target);

	iov[0].iov_base = domid_str;
	iov[0].iov_len = strlen(domid_str) + 1;
	iov[1].iov_base = target_str;
	iov[1].iov_len = strlen(target_str) + 1;

	return xs_bool(xs_talkv(h, XBT_NULL, XS_SET_TARGET, iov,
				ARRAY_SIZE(iov), NULL));
}

static void * single_with_domid(struct xs_handle *h,
				enum xsd_sockmsg_type type,
				unsigned int domid)
{
	char domid_str[MAX_STRLEN(domid)];

	snprintf(domid_str, sizeof(domid_str), "%u", domid);

	return xs_single(h, XBT_NULL, type, domid_str, NULL);
}

bool xs_release_domain(struct xs_handle *h, unsigned int domid)
{
	return xs_bool(single_with_domid(h, XS_RELEASE, domid));
}

/* clear the shutdown bit for the given domain */
bool xs_resume_domain(struct xs_handle *h, unsigned int domid)
{
	return xs_bool(single_with_domid(h, XS_RESUME, domid));
}

char *xs_get_domain_path(struct xs_handle *h, unsigned int domid)
{
	char domid_str[MAX_STRLEN(domid)];

	snprintf(domid_str, sizeof(domid_str), "%u", domid);

	return xs_single(h, XBT_NULL, XS_GET_DOMAIN_PATH, domid_str, NULL);
}

bool xs_is_domain_introduced(struct xs_handle *h, unsigned int domid)
{
	char *domain = single_with_domid(h, XS_IS_DOMAIN_INTRODUCED, domid);
	int rc = strcmp("F", domain);

	free(domain);
	return rc;
}

/* Only useful for DEBUG versions */
char *xs_debug_command(struct xs_handle *h, const char *cmd,
		       void *data, unsigned int len)
{
	struct iovec iov[2];

	iov[0].iov_base = (void *)cmd;
	iov[0].iov_len = strlen(cmd) + 1;
	iov[1].iov_base = data;
	iov[1].iov_len = len;

	return xs_talkv(h, XBT_NULL, XS_DEBUG, iov,
			ARRAY_SIZE(iov), NULL);
}

static int read_message(struct xs_handle *h)
{
	struct xs_stored_msg *msg = NULL;
	char *body = NULL;
	int saved_errno;

	/* Allocate message structure and read the message header. */
	msg = malloc(sizeof(*msg));
	if (msg == NULL)
		goto error;
	if (!read_all(h->fd, &msg->hdr, sizeof(msg->hdr)))
		goto error;

	/* Allocate and read the message body. */
	body = msg->body = malloc(msg->hdr.len + 1);
	if (body == NULL)
		goto error;
	if (!read_all(h->fd, body, msg->hdr.len))
		goto error;
	body[msg->hdr.len] = '\0';

	if (msg->hdr.type == XS_WATCH_EVENT) {
		mutex_lock(&h->watch_mutex);

		/* Kick users out of their select() loop. */
		if (list_empty(&h->watch_list) &&
		    (h->watch_pipe[1] != -1))
			while (write(h->watch_pipe[1], body, 1) != 1)
				continue;

		list_add_tail(&msg->list, &h->watch_list);

		condvar_signal(&h->watch_condvar);

		mutex_unlock(&h->watch_mutex);
	} else {
		mutex_lock(&h->reply_mutex);

		/* There should only ever be one response pending! */
		if (!list_empty(&h->reply_list)) {
			mutex_unlock(&h->reply_mutex);
			goto error;
		}

		list_add_tail(&msg->list, &h->reply_list);
		condvar_signal(&h->reply_condvar);

		mutex_unlock(&h->reply_mutex);
	}

	return 0;

 error:
	saved_errno = errno;
	free(msg);
	free(body);
	errno = saved_errno;
	return -1;
}

#ifdef USE_PTHREAD
static void *read_thread(void *arg)
{
	struct xs_handle *h = arg;

	while (read_message(h) != -1)
		continue;

	return NULL;
}
#endif

/*
 * Local variables:
 *  c-file-style: "linux"
 *  indent-tabs-mode: t
 *  c-indent-level: 8
 *  c-basic-offset: 8
 *  tab-width: 8
 * End:
 */
                                                                                                                                    xenstore/xs.h                                                                                       0000644 0001750 0001750 00000013175 11146403776 013110  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /* 
    Xen Store Daemon providing simple tree-like database.
    Copyright (C) 2005 Rusty Russell IBM Corporation

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _XS_H
#define _XS_H

#include <xs_lib.h>

#define XBT_NULL 0

struct xs_handle;
typedef uint32_t xs_transaction_t;

/* On failure, these routines set errno. */

/* Connect to the xs daemon.
 * Returns a handle or NULL.
 */
struct xs_handle *xs_daemon_open(void);
struct xs_handle *xs_domain_open(void);

/* Connect to the xs daemon (readonly for non-root clients).
 * Returns a handle or NULL.
 */
struct xs_handle *xs_daemon_open_readonly(void);

/* Close the connection to the xs daemon. */
void xs_daemon_close(struct xs_handle *);

/* Get contents of a directory.
 * Returns a malloced array: call free() on it after use.
 * Num indicates size.
 */
char **xs_directory(struct xs_handle *h, xs_transaction_t t,
		    const char *path, unsigned int *num);

/* Get the value of a single file, nul terminated.
 * Returns a malloced value: call free() on it after use.
 * len indicates length in bytes, not including terminator.
 */
void *xs_read(struct xs_handle *h, xs_transaction_t t,
	      const char *path, unsigned int *len);

/* Write the value of a single file.
 * Returns false on failure.
 */
bool xs_write(struct xs_handle *h, xs_transaction_t t,
	      const char *path, const void *data, unsigned int len);

/* Create a new directory.
 * Returns false on failure, or success if it already exists.
 */
bool xs_mkdir(struct xs_handle *h, xs_transaction_t t,
	      const char *path);

/* Destroy a file or directory (and children).
 * Returns false on failure, or if it doesn't exist.
 */
bool xs_rm(struct xs_handle *h, xs_transaction_t t,
	   const char *path);

/* Get permissions of node (first element is owner, first perms is "other").
 * Returns malloced array, or NULL: call free() after use.
 */
struct xs_permissions *xs_get_permissions(struct xs_handle *h,
					  xs_transaction_t t,
					  const char *path, unsigned int *num);

/* Set permissions of node (must be owner).
 * Returns false on failure.
 */
bool xs_set_permissions(struct xs_handle *h, xs_transaction_t t,
			const char *path, struct xs_permissions *perms,
			unsigned int num_perms);

/* Watch a node for changes (poll on fd to detect, or call read_watch()).
 * When the node (or any child) changes, fd will become readable.
 * Token is returned when watch is read, to allow matching.
 * Returns false on failure.
 */
bool xs_watch(struct xs_handle *h, const char *path, const char *token);

/* Return the FD to poll on to see if a watch has fired. */
int xs_fileno(struct xs_handle *h);

/* Find out what node change was on (will block if nothing pending).
 * Returns array containing the path and token. Use XS_WATCH_* to access these
 * elements. Call free() after use.
 */
char **xs_read_watch(struct xs_handle *h, unsigned int *num);

/* Remove a watch on a node: implicitly acks any outstanding watch.
 * Returns false on failure (no watch on that node).
 */
bool xs_unwatch(struct xs_handle *h, const char *path, const char *token);

/* Start a transaction: changes by others will not be seen during this
 * transaction, and changes will not be visible to others until end.
 * Returns NULL on failure.
 */
xs_transaction_t xs_transaction_start(struct xs_handle *h);

/* End a transaction.
 * If abandon is true, transaction is discarded instead of committed.
 * Returns false on failure: if errno == EAGAIN, you have to restart
 * transaction.
 */
bool xs_transaction_end(struct xs_handle *h, xs_transaction_t t,
			bool abort);

/* Introduce a new domain.
 * This tells the store daemon about a shared memory page, event channel and
 * store path associated with a domain: the domain uses these to communicate.
 */
bool xs_introduce_domain(struct xs_handle *h,
			 unsigned int domid,
			 unsigned long mfn,
                         unsigned int eventchn); 

/* Set the target of a domain
 * This tells the store daemon that a domain is targetting another one, so
 * it should let it tinker with it.
 */
bool xs_set_target(struct xs_handle *h,
		   unsigned int domid,
		   unsigned int target);

/* Resume a domain.
 * Clear the shutdown flag for this domain in the store.
 */
bool xs_resume_domain(struct xs_handle *h, unsigned int domid);

/* Release a domain.
 * Tells the store domain to release the memory page to the domain.
 */
bool xs_release_domain(struct xs_handle *h, unsigned int domid);

/* Query the home path of a domain.  Call free() after use.
 */
char *xs_get_domain_path(struct xs_handle *h, unsigned int domid);

/* Return whether the domain specified has been introduced to xenstored.
 */
bool xs_is_domain_introduced(struct xs_handle *h, unsigned int domid);

/* Only useful for DEBUG versions */
char *xs_debug_command(struct xs_handle *h, const char *cmd,
		       void *data, unsigned int len);

#endif /* _XS_H */

/*
 * Local variables:
 *  c-file-style: "linux"
 *  indent-tabs-mode: t
 *  c-indent-level: 8
 *  c-basic-offset: 8
 *  tab-width: 8
 * End:
 */
                                                                                                                                                                                                                                                                                                                                                                                                   xenstore/xs_lib.c                                                                                   0000644 0001750 0001750 00000014141 11146403776 013723  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /* 
    Common routines between Xen store user library and daemon.
    Copyright (C) 2005 Rusty Russell IBM Corporation

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include <assert.h>
#include "xs_lib.h"

/* Common routines for the Xen store daemon and client library. */

const char *xs_daemon_rootdir(void)
{
	char *s = getenv("XENSTORED_ROOTDIR");
	return (s ? s : "/var/lib/xenstored");
}

const char *xs_daemon_rundir(void)
{
	char *s = getenv("XENSTORED_RUNDIR");
	return (s ? s : "/var/run/xenstored");
}

static const char *xs_daemon_path(void)
{
	static char buf[PATH_MAX];
	char *s = getenv("XENSTORED_PATH");
	if (s)
		return s;
	if (snprintf(buf, sizeof(buf), "%s/socket",
		     xs_daemon_rundir()) >= PATH_MAX)
		return NULL;
	return buf;
}

const char *xs_daemon_tdb(void)
{
	static char buf[PATH_MAX];
	snprintf(buf, sizeof(buf), "%s/tdb", xs_daemon_rootdir());
	return buf;
}

const char *xs_daemon_socket(void)
{
	return xs_daemon_path();
}

const char *xs_daemon_socket_ro(void)
{
	static char buf[PATH_MAX];
	const char *s = xs_daemon_path();
	if (s == NULL)
		return NULL;
	if (snprintf(buf, sizeof(buf), "%s_ro", s) >= PATH_MAX)
		return NULL;
	return buf;
}

const char *xs_domain_dev(void)
{
	char *s = getenv("XENSTORED_PATH");
	if (s)
		return s;

#if defined(__linux__)
	return "/proc/xen/xenbus";
#elif defined(__NetBSD__)
	return "/kern/xen/xenbus";
#else
	return "/dev/xen/xenbus";
#endif
}

/* Simple routines for writing to sockets, etc. */
bool xs_write_all(int fd, const void *data, unsigned int len)
{
	while (len) {
		int done;

		done = write(fd, data, len);
		if (done < 0 && errno == EINTR)
			continue;
		if (done <= 0)
			return false;
		data += done;
		len -= done;
	}

	return true;
}

/* Convert strings to permissions.  False if a problem. */
bool xs_strings_to_perms(struct xs_permissions *perms, unsigned int num,
			 const char *strings)
{
	const char *p;
	char *end;
	unsigned int i;

	for (p = strings, i = 0; i < num; i++) {
		/* "r", "w", or "b" for both. */
		switch (*p) {
		case 'r':
			perms[i].perms = XS_PERM_READ;
			break;
		case 'w':
			perms[i].perms = XS_PERM_WRITE;
			break;
		case 'b':
			perms[i].perms = XS_PERM_READ|XS_PERM_WRITE;
			break;
		case 'n':
			perms[i].perms = XS_PERM_NONE;
			break;
		default:
			errno = EINVAL;
			return false;
		} 
		p++;
		perms[i].id = strtol(p, &end, 0);
		if (*end || !*p) {
			errno = EINVAL;
			return false;
		}
		p = end + 1;
	}
	return true;
}

/* Convert permissions to a string (up to len MAX_STRLEN(unsigned int)+1). */
bool xs_perm_to_string(const struct xs_permissions *perm,
                       char *buffer, size_t buf_len)
{
	switch (perm->perms) {
	case XS_PERM_WRITE:
		*buffer = 'w';
		break;
	case XS_PERM_READ:
		*buffer = 'r';
		break;
	case XS_PERM_READ|XS_PERM_WRITE:
		*buffer = 'b';
		break;
	case XS_PERM_NONE:
		*buffer = 'n';
		break;
	default:
		errno = EINVAL;
		return false;
	}
	snprintf(buffer+1, buf_len-1, "%i", (int)perm->id);
	return true;
}

/* Given a string and a length, count how many strings (nul terms). */
unsigned int xs_count_strings(const char *strings, unsigned int len)
{
	unsigned int num;
	const char *p;

	for (p = strings, num = 0; p < strings + len; p++)
		if (*p == '\0')
			num++;

	return num;
}

char *expanding_buffer_ensure(struct expanding_buffer *ebuf, int min_avail)
{
	int want;
	char *got;

	if (ebuf->avail >= min_avail)
		return ebuf->buf;

	if (min_avail >= INT_MAX/3)
		return 0;

	want = ebuf->avail + min_avail + 10;
	got = realloc(ebuf->buf, want);
	if (!got)
		return 0;

	ebuf->buf = got;
	ebuf->avail = want;
	return ebuf->buf;
}

char *sanitise_value(struct expanding_buffer *ebuf,
		     const char *val, unsigned len)
{
	int used, remain, c;
	unsigned char *ip;

#define ADD(c) (ebuf->buf[used++] = (c))
#define ADDF(f,c) (used += sprintf(ebuf->buf+used, (f), (c)))

	assert(len < INT_MAX/5);

	ip = (unsigned char *)val;
	used = 0;
	remain = len;

	if (!expanding_buffer_ensure(ebuf, remain + 1))
		return NULL;

	while (remain-- > 0) {
		c= *ip++;

		if (c >= ' ' && c <= '~' && c != '\\') {
			ADD(c);
			continue;
		}

		if (!expanding_buffer_ensure(ebuf, used + remain + 5))
			/* for "<used>\\nnn<remain>\0" */
			return 0;

		ADD('\\');
		switch (c) {
		case '\t':  ADD('t');   break;
		case '\n':  ADD('n');   break;
		case '\r':  ADD('r');   break;
		case '\\':  ADD('\\');  break;
		default:
			if (c < 010) ADDF("%03o", c);
			else         ADDF("x%02x", c);
		}
	}

	ADD(0);
	assert(used <= ebuf->avail);
	return ebuf->buf;

#undef ADD
#undef ADDF
}

void unsanitise_value(char *out, unsigned *out_len_r, const char *in)
{
	const char *ip;
	char *op;
	unsigned c;
	int n;

	for (ip = in, op = out; (c = *ip++); *op++ = c) {
		if (c == '\\') {
			c = *ip++;

#define GETF(f) do {					\
		        n = 0;				\
                        sscanf(ip, f "%n", &c, &n);	\
			ip += n;			\
		} while (0)

			switch (c) {
			case 't':              c= '\t';            break;
			case 'n':              c= '\n';            break;
			case 'r':              c= '\r';            break;
			case '\\':             c= '\\';            break;
			case 'x':                    GETF("%2x");  break;
			case '0': case '4':
			case '1': case '5':
			case '2': case '6':
			case '3': case '7':    --ip; GETF("%3o");  break;
			case 0:                --ip;               break;
			default:;
			}
#undef GETF
		}
	}

	*op = 0;

	if (out_len_r)
		*out_len_r = op - out;
}
                                                                                                                                                                                                                                                                                                                                                                                                                               xenstore/xs_lib.h                                                                                   0000644 0001750 0001750 00000005451 11146403776 013734  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 /* 
    Common routines between Xen store user library and daemon.
    Copyright (C) 2005 Rusty Russell IBM Corporation

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _XS_LIB_H
#define _XS_LIB_H

#include <stdbool.h>
#include <limits.h>
#include <errno.h>
#include <stdint.h>
#include <xen/io/xs_wire.h>

/* Bitmask of permissions. */
enum xs_perm_type {
	XS_PERM_NONE = 0,
	XS_PERM_READ = 1,
	XS_PERM_WRITE = 2,
	/* Internal use. */
	XS_PERM_ENOENT_OK = 4,
	XS_PERM_OWNER = 8,
};

struct xs_permissions
{
	unsigned int id;
	enum xs_perm_type perms;
};

/* Each 10 bits takes ~ 3 digits, plus one, plus one for nul terminator. */
#define MAX_STRLEN(x) ((sizeof(x) * CHAR_BIT + CHAR_BIT-1) / 10 * 3 + 2)

/* Path for various daemon things: env vars can override. */
const char *xs_daemon_rootdir(void);
const char *xs_daemon_rundir(void);
const char *xs_daemon_socket(void);
const char *xs_daemon_socket_ro(void);
const char *xs_domain_dev(void);
const char *xs_daemon_tdb(void);

/* Simple write function: loops for you. */
bool xs_write_all(int fd, const void *data, unsigned int len);

/* Convert strings to permissions.  False if a problem. */
bool xs_strings_to_perms(struct xs_permissions *perms, unsigned int num,
			 const char *strings);

/* Convert permissions to a string (up to len MAX_STRLEN(unsigned int)+1). */
bool xs_perm_to_string(const struct xs_permissions *perm,
                       char *buffer, size_t buf_len);

/* Given a string and a length, count how many strings (nul terms). */
unsigned int xs_count_strings(const char *strings, unsigned int len);

/* Sanitising (quoting) possibly-binary strings. */
struct expanding_buffer {
	char *buf;
	int avail;
};

/* Ensure that given expanding buffer has at least min_avail characters. */
char *expanding_buffer_ensure(struct expanding_buffer *, int min_avail);

/* sanitise_value() may return NULL if malloc fails. */
char *sanitise_value(struct expanding_buffer *, const char *val, unsigned len);

/* *out_len_r on entry is ignored; out must be at least strlen(in)+1 bytes. */
void unsanitise_value(char *out, unsigned *out_len_r, const char *in);

#endif /* _XS_LIB_H */
                                                                                                                                                                                                                       xenstore/xs.ml                                                                                      0000644 0001750 0001750 00000000421 11146404050 013241  0                                                                                                    ustar   pjcolp                          pjcolp                                                                                                                                                                                                                 let xs_single connection message_type transaction_id payload =
  let message = Message.make message_type transaction_id 0l payload in
  connection#write message;;

let rec xs_read connection =
  match (connection#read) with
  | Some m -> m
  | None -> xs_read connection;;
                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               