/**************************************************************************\
*//*! \file ef_char_bend.c Xen back end driver/EF resources

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/


#include "ci/driver/virtual/ef_hyperops.h"
#include <ci/driver/efab/internal.h>
#include <ci/internal/pt.h>
#include <ci/ul/resource_init.h>
#include <ci/driver/efab/driver_object.h>
#include <ci/driver/platform/linux_pci_compat.h>
#include "ef_bend.h"
#include "ef_filter.h"
#include "ef_char_bend.h"


/****************************************************************************
 * Linkage support
 ***************************************************************************/

/* Self-linkage support: to avoid a load-time dependency on the
 * char driver (which won't be loaded if there is no hardware 
 * present) we resolve the symbols we need by hand at initialisation.
 * The main ugliness is that some char driver symbols are refenced from
 * inline function. To solve that we have stubs of the same name that 
 * indirect through the function pointers to the real functions. This
 * works because symbol_get will not find the stubs (they are not exports)
 * so we do not end up looping: the function pointers point to the non-stub
 * versions. */
ci_driver_t *ptr_ci_driver;
#define EF_FUNC_PTR(_name) typeof(&_name) fn_ ##_name
EF_FUNC_PTR(ef_vi_init);
EF_FUNC_PTR(ef_iobufset_alloc);
EF_FUNC_PTR(ef_iobufset_free);
EF_FUNC_PTR(ef_eventq_initialize_for_one_nic);
EF_FUNC_PTR(ef_eventq_free);
EF_FUNC_PTR(ef_eventq_allocate_hardware_resource_for_all_nics);
EF_FUNC_PTR(efab_eventq_register_callback);
EF_FUNC_PTR(efab_eventq_kill_callback);
EF_FUNC_PTR(efab_resource_manager_alloc);
EF_FUNC_PTR(efab_filter_resource_clear);
EF_FUNC_PTR(efab_resource_assert_valid);
EF_FUNC_PTR(efab_resource_manager_assert_valid);
EF_FUNC_PTR(__efab_filter_resource_set);
EF_FUNC_PTR(__efab_resource_ref_count_zero);
EF_FUNC_PTR(efab_eventq_reset);
EF_FUNC_PTR(efab_buffer_table_alloc);
EF_FUNC_PTR(efab_pci_map_single);
EF_FUNC_PTR(efab_buffer_table_set);
EF_FUNC_PTR(efab_buffer_table_commit);
EF_FUNC_PTR(ef_eventq_id);

/* Initialise the char driver symbols */
int ef_bend_char_init(void)
{
#define try_symbol_get(_pfx, _name)  \
  do {                               \
    _pfx##_name = symbol_get(_name); \
    if(_pfx##_name == NULL)          \
      goto bad_symbol;               \
  } while(0)

  try_symbol_get(ptr_, ci_driver);
  try_symbol_get(fn_, ef_eventq_free);
  try_symbol_get(fn_, ef_eventq_initialize_for_one_nic);
  try_symbol_get(fn_, ef_eventq_allocate_hardware_resource_for_all_nics);
  try_symbol_get(fn_, efab_eventq_register_callback);
  try_symbol_get(fn_, efab_eventq_kill_callback);
  try_symbol_get(fn_, ef_iobufset_alloc);
  try_symbol_get(fn_, ef_iobufset_free);
  try_symbol_get(fn_, ef_vi_init);
  try_symbol_get(fn_, efab_resource_manager_alloc);
  try_symbol_get(fn_, efab_filter_resource_clear);
  try_symbol_get(fn_, efab_resource_assert_valid);
  try_symbol_get(fn_, efab_resource_manager_assert_valid);
  try_symbol_get(fn_, __efab_filter_resource_set);
  try_symbol_get(fn_, __efab_resource_ref_count_zero);
  try_symbol_get(fn_, efab_eventq_reset);
  try_symbol_get(fn_, efab_buffer_table_alloc);
  try_symbol_get(fn_, efab_pci_map_single);
  try_symbol_get(fn_, efab_buffer_table_set);
  try_symbol_get(fn_, efab_buffer_table_commit);
  try_symbol_get(fn_, ef_eventq_id);
#undef try_symbol_get

  return 0;

bad_symbol:
  ef_bend_char_shutdown();
  return -ENOENT;
}

void ef_bend_char_shutdown(void)
{
#define maybe_symbol_put(_pfx,_name) \
  do {                               \
    if(_pfx##_name != NULL) {        \
      symbol_put(_name);             \
      _pfx##_name = NULL;            \
    }                                \
  } while(0)

  maybe_symbol_put(ptr_,ci_driver);
  maybe_symbol_put(fn_, ef_eventq_free);
  maybe_symbol_put(fn_, ef_eventq_initialize_for_one_nic);
  maybe_symbol_put(fn_, ef_eventq_allocate_hardware_resource_for_all_nics);
  maybe_symbol_put(fn_, efab_eventq_register_callback);
  maybe_symbol_put(fn_, efab_eventq_kill_callback);
  maybe_symbol_put(fn_, ef_iobufset_alloc);
  maybe_symbol_put(fn_, ef_iobufset_free);
  maybe_symbol_put(fn_, ef_vi_init);
  maybe_symbol_put(fn_, efab_resource_manager_alloc);
  maybe_symbol_put(fn_, efab_filter_resource_clear);
  maybe_symbol_put(fn_, efab_resource_assert_valid);
  maybe_symbol_put(fn_, efab_resource_manager_assert_valid);
  maybe_symbol_put(fn_, __efab_filter_resource_set);
  maybe_symbol_put(fn_, __efab_resource_ref_count_zero);
  maybe_symbol_put(fn_, efab_eventq_reset);
  maybe_symbol_put(fn_, efab_buffer_table_alloc);
  maybe_symbol_put(fn_, efab_pci_map_single);
  maybe_symbol_put(fn_, efab_buffer_table_set);
  maybe_symbol_put(fn_, efab_buffer_table_commit);
  maybe_symbol_put(fn_, ef_eventq_id);
#undef maybe_symbol_put
}

/* Stubs required (see explanation above) */
void efab_resource_assert_valid(efab_resource_t *res, int rc_may_be_zero,
                                const char *file, int line)
{
  fn_efab_resource_assert_valid(res, rc_may_be_zero, file, line);
}

void efab_resource_manager_assert_valid(efab_resource_manager_t *rm,
                                      const char *file, int line)
{
  fn_efab_resource_manager_assert_valid(rm, file, line);
}

int __efab_filter_resource_set(filter_resource_t* frs, int type, 
                               unsigned saddr_be32, ci_uint16 sport_be16,
                               unsigned daddr_be32, ci_uint16 dport_be16)
{
  return fn___efab_filter_resource_set(frs, type, saddr_be32, sport_be16,
                                       daddr_be32, dport_be16);
}

void __efab_resource_ref_count_zero(efab_resource_manager_t* rm,
                                    unsigned instance)
{
  fn___efab_resource_ref_count_zero(rm, instance);
}

/****************************************************************************
 * Resource management code
 ***************************************************************************/

/* This is basically allocate_pt_endpoint without the dependencies */
static int alloc_pt(efab_resource_t **rs, ef_vi *vi, ef_eventq *evq, int nic_index, int use_phys_addr)
{
    efab_resource_manager_t* rm;
    ci_resource_alloc_t ra; 
    unsigned instance;
    int rc;
    void *tmp = kzalloc(EF_VI_STATE_BYTES, GFP_KERNEL);
    rm = ptr_ci_driver->rm[EFAB_RESOURCE_PT_ENDPOINT];
    ci_assert(rm);
  
    ci_resource_alloc_init(&ra, rm->rm_type);
    ra.u.pt.in_zero = 0;
    CI_USER_PTR_SET(ra.u.pt.in_evq.rs, evq->evq_mmap.rs);
    ra.u.pt.in_rx_q_tag = 0;
    ra.u.pt.in_tx_q_tag = 0;
    ra.u.pt.in_flags = 0;
    if(use_phys_addr) {
      ra.u.pt.in_flags |= EFAB_VI_PHYS_ADDR_EN;
    }
    ra.u.pt.in_for_iscsi = CI_FALSE;
    rc = fn_efab_resource_manager_alloc(rm, 0, &ra, rs);
    if( rc < 0 ) {
      ci_log("%s: ERROR efab_resource_manager_alloc error %d", __FUNCTION__, rc);
      kfree(tmp);
      return rc;
    }

    instance = EFAB_RESOURCE_INSTANCE( (*rs)->rs_handle);
    ci_log("Got back a PT endpoint resource.  Instance %u",
        instance);
    fn_ef_vi_init(vi, &ptr_ci_driver->nic[0]->bar[0],
               ci_mk_mmap_info(*rs, ra.ra_mmap.bytes), 0, 0,
               (ef_vi_state*) tmp);
    kfree(tmp);
    return rc;
}


/* Allocate the eventq */
int alloc_evq(ef_eventq *evq, eventq_resource_t **rs, int capacity, 
              int nic_index, int use_phys_addr)
{
  ci_uint32 mmap_bytes;
  ef_eventq_state dummy;

  int rc = fn_ef_eventq_allocate_hardware_resource_for_all_nics
    (capacity, use_phys_addr, CI_FALSE, rs, &mmap_bytes);

  if( rc < 0 ) {
    ci_log("%s: efab_resource_manager_alloc eventq failure code %d",
           __FUNCTION__, rc);
    return rc;
  } 
  fn_ef_eventq_initialize_for_one_nic(*rs, mmap_bytes, evq, nic_index, &dummy);

  ci_log("Event q buffer is order %d, at kva %p and dma addr " ci_dma_addr_fmt "\n"
         "offset %x capacity %d",
         (*rs)->hardware[0].iobuff.order, 
         (*rs)->hardware[0].iobuff.kva, 
         (*rs)->hardware[0].iobuff.dma_addr,
         (*rs)->hardware[0].iobuff_off,
         capacity);

  return 0;
}


/* Create an EFAB filter resource, return in fres_out if no error (rc
   = 0) */
int ef_char_alloc_filter(efab_resource_handle_t pthandle,
                        filter_resource_t **fres_out)
{
  efab_resource_manager_t * frm;
  efab_resource_t * rs;
  ci_resource_alloc_t fra;
  int rc;
  frm = ptr_ci_driver->rm[EFAB_RESOURCE_FILTER];
  ci_assert(frm);
  ci_resource_alloc_init(&fra, EFAB_RESOURCE_FILTER);
  fra.u.filter.in_pt_handle = pthandle;
  BEND_VERB(ci_log("alloc_filter called with pthandle %x", pthandle.handle));
  rc = fn_efab_resource_manager_alloc(frm, 0, &fra, &rs);
  if (rc < 0) {
    *fres_out = NULL;
    return rc;
  }
  *fres_out = CI_CONTAINER(filter_resource_t, rs, rs);
  return 0;
}

void ef_char_free_filter(filter_resource_t *fres)
{
  ci_assert(fres);
  efab_resource_release(&fres->rs);
}

/* The timeout event callback for the event q */
static void bend_evq_timeout(ci_uintptr_t context, 
                             int is_timeout, struct efab_nic_s *nic)
{
  struct ef_bend *bend = (struct ef_bend *)context;
  if (is_timeout) {
    /* Pass event to vnic front end driver */
    BEND_VERB(ci_log("timeout event to %d", bend->net_irq));
    ef_hyperop_remote_irq(bend->net_irq);
  } else {
    /* Probably a wakeup event, but we don't request wakeups at the
       moment */
    ci_log("*** Why is this happening to me?");
  }
}

/* Create the eventq and associated gubbins for communication with the
   front end vnic driver */
int ef_get_vnic(struct ef_bend *bend)
{
  int eventqsize = CI_CFG_NETIF_EVENTQ_SIZE;
  int rc = 0;
  ef_vi *vi;

  /* This may already have been set up. */
  if (bend->hw_state == EF_RES_ALLOC) {
    eventq_resource_t *evq_res = bend->ep_evq_res;
    ci_log("%s: bend %p already has H/W. Not reallocating", __FUNCTION__, bend);
    fn_efab_eventq_reset(evq_res, 0);
    /* We would have removed this when the other end shut down. */
    fn_efab_eventq_register_callback((eventq_resource_t *)bend->ep_evq_res, 
                                      bend_evq_timeout, (ci_uintptr_t)bend);

    return rc;
  }


  vi = kzalloc(sizeof(ef_vi), GFP_KERNEL);

  /* Fill out bend structure, and allocate some memory for the
     eventq */
  bend->ep_vi = vi;
  bend->ep_evq = kzalloc(sizeof(struct ef_eventq_struct), GFP_KERNEL);
  if (bend->ep_evq == NULL)
    return -ENOMEM;
  rc = alloc_evq(bend->ep_evq,  (eventq_resource_t **)&bend->ep_evq_res, 
                 eventqsize, 0, 0);
  if (rc) {
    ci_log("alloc_evq failed: %d", rc);
    goto fail_no_evq;
  }

  /* basically allocate_pt_endpoint() */
  rc = alloc_pt(&bend->ep_res, vi, bend->ep_evq, 0,0);
  if (rc) {
    ci_log("alloc_pt failed: %d", rc);
    goto fail_no_pt;
  }


  /* Register the eventq timeout event callback */
  fn_efab_eventq_register_callback((eventq_resource_t *)bend->ep_evq_res, 
                                 bend_evq_timeout, (ci_uintptr_t)bend);


  return 0;

fail_no_pt:
  fn_ef_eventq_free(bend->ep_evq, 0);
  kfree(bend->ep_evq);
  bend->ep_evq = NULL;
fail_no_evq:
  return rc;
}

void ef_free_vnic(struct ef_bend *bend)
{
  eventq_resource_t *evqrs = bend->ep_evq_res;
  ef_vi *vi = bend->ep_vi;

  if (evqrs->callback_fn) {
    /* Disable callbacks */
    fn_efab_eventq_kill_callback(evqrs);
  }


  bend->num_bufs = 0;

  /* Either it was never allocated, or it is not safe to release */
  if (bend->hw_state != EF_RES_DONE) {
    ci_log("Cannot free hardware, reason: %s",
           bend->hw_state ? "in use" : "never allocated");
    /* It may be reused if the domain that had it reloads the driver,
     * so reset its state. */
    if (bend->hw_state == EF_RES_ALLOC) {
      int evqnum = EFAB_RESOURCE_INSTANCE(evqrs->rs.rs_handle);
      /* Throw away any pending DMA */
      efab_nic_flush_tx_dma_channel(ptr_ci_driver->nic[0],
                                    vi->ep_dma_tx_q.dmaq);
      efab_nic_flush_rx_dma_channel(ptr_ci_driver->nic[0],
                                    vi->ep_dma_rx_q.dmaq);
      ci_log("Reinit DMA Qs");
      efab_nic_dmaq_tx_q_init(ptr_ci_driver->nic[0], vi->ep_dma_tx_q.dmaq,
                              evqnum, evqnum, 0, 0 /* FIXME falcon */,
                              0, 0);
      efab_nic_dmaq_rx_q_init(ptr_ci_driver->nic[0], vi->ep_dma_rx_q.dmaq,
                              evqnum, evqnum, 0, 0 /* FIXME falcon */,
                              0, 0);
    }
    return;
  }
  ci_log("Hardware is freeable. Will proceed.");
  /* Free the NIC buffers */
  if (bend->iobufs) {
    fn_ef_iobufset_free(bend->iobufs, 0);
    kfree(bend->iobufs);
    bend->iobufs = NULL;
  }

  /* And the event queue */
  if (bend->ep_evq) {
    fn_ef_eventq_free(bend->ep_evq, 0);
    kfree(bend->ep_evq);
    bend->ep_evq = NULL;
  }
  /* And the VNIC */
  if (bend->ep_res) {
    ci_log("Freeing pt resource (ref. count %d)", bend->ep_res->rs_ref_count.n);
    efab_resource_release(bend->ep_res);
    kfree(bend->ep_vi);
    bend->ep_vi = NULL;
  }
}

/* Find the hardware pages for the various resources we have allocated. This 
  * is a tad ugly because the resource manager doesn't generally expose physical
  * addresses, having done the mapping itself.  Sets up state in ef_msg_hw for 
  * a number of useful things at various offsets within these pages, such that
  * the vnic can reconstruct working DMA Qs, event Qs, etc. from the info. in
  * the message. */

static void ef_bend_hwinfo_ef1(struct ef_bend *bend, struct ef_hw_ef1 *hwinfo)
{
  unsigned instance = EFAB_RESOURCE_INSTANCE( bend->ep_res->rs_handle);

  efab_nic_t* nic = ptr_ci_driver->nic[0];
  ef_vi *vi = bend->ep_vi;
  eventq_resource_t *evqr = bend->ep_evq_res;
  ef_eventq *evq = bend->ep_evq;
  int evqinstance = EFAB_RESOURCE_INSTANCE(evqr->rs.rs_handle);
  int i, npages;
  int pfn;
  void *evq_mem_kva;        /*!< KVA of queue. */

  ci_phys_addr_t phys = nic->os.ln.pci_addr[EF1002_PAGED_VNIC_BAR];
  ci_phys_addr_t etphys = nic->os.ln.pci_addr[EF1002_CTR_AP0_BAR];

  hwinfo->txdmaq = 
      (phys + EF1002_PAGED_VNIC_TXDMAQ_OFF + EF1002_PAGED_RESOURCE_BYTES * instance) >> PAGE_SHIFT;
  hwinfo->rxdmaq = 
      (phys + EF1002_PAGED_VNIC_RXDMAQ_OFF + EF1002_PAGED_RESOURCE_BYTES * instance) >> PAGE_SHIFT;
  hwinfo->txbell = 
      (phys + EF1002_PAGED_VNIC_TXBELL_OFF + EF1002_PAGED_RESOURCE_BYTES * instance) >> PAGE_SHIFT;
  hwinfo->rxbell = 
      (phys + EF1002_PAGED_VNIC_RXBELL_OFF + EF1002_PAGED_RESOURCE_BYTES * instance) >> PAGE_SHIFT;
  hwinfo->txid = vi->ep_dma_tx_q.dmaq;
  hwinfo->rxid = vi->ep_dma_rx_q.dmaq;

  hwinfo->evq_order = evqr->hardware[0].iobuff.order;
  hwinfo->evq_offs = evqr->hardware[0].iobuff_off;
  hwinfo->evq_capacity = CI_CFG_NETIF_EVENTQ_SIZE;
  evq_mem_kva = evqr->hardware[0].iobuff.kva;

  hwinfo->evq_timer = etphys +  ef1002_timer_page_addr(evqinstance);
  ci_log("evq_timer %d is at %p", evqinstance, evq->evq_timer_reg);

  hwinfo->evq_ptr = etphys +  EF1002_EVENTQ_PTR_OFF +
    (EF1002_EVENTQ_PTR_BYTES * evqinstance);

  /* Make the relevant H/W pages mappable by the far end */
  ef_hyperop_pass_io_page(bend->hdev_data, "txq-page", hwinfo->txdmaq,
                          bend->far_end, EF_HYPEROP_READWRITE);
  ef_hyperop_pass_io_page(bend->hdev_data, "rxq-page", hwinfo->rxdmaq,
                          bend->far_end, EF_HYPEROP_READWRITE);
  ef_hyperop_pass_io_page(bend->hdev_data, "txbell-page", hwinfo->txbell,
                          bend->far_end, EF_HYPEROP_READWRITE);
  ef_hyperop_pass_io_page(bend->hdev_data, "rxbell-page", hwinfo->rxbell,
                          bend->far_end, EF_HYPEROP_READWRITE);
  /* Note that unlike the above, where the message field is the page number, here
   * evq_timer is the entire address because it is currently a pointer into the
   * densely mapped timer page. */
  ef_hyperop_pass_io_page(bend->hdev_data, "evq-timer",
                          hwinfo->evq_timer >> PAGE_SHIFT, bend->far_end, 
                          EF_HYPEROP_READWRITE);
  ef_hyperop_pass_io_page(bend->hdev_data, "evq-ptr", 
                          hwinfo->evq_ptr >> PAGE_SHIFT, bend->far_end,
                          EF_HYPEROP_READONLY);

  /* Now do the same for the memory pages */
  /* Convert the page + length we got back for the evq to grants. */
  npages = 1 << hwinfo->evq_order;
  pfn = page_to_pfn(virt_to_page(evq_mem_kva));

  for (i = 0; i < npages; i++) {
    hwinfo->evq_mem.gnts[i] = ef_hyperop_pass_mem_page(bend->hdev_data, 
                                                      "evq page", pfn,
                                                       bend->far_end);
    BEND_VERB(ci_log("Got grant %u for evq pfn %u", hwinfo->evq_mem.gnts[i], 
              pfn));
    pfn++;
  }

  hwinfo->phys_port = bend->phys_port;
}

/* Fill in the message with a description of the hardware resources, based on
 * the H/W type */
int ef_bend_hwinfo(struct ef_bend *bend, struct ef_msg_hw *msgvi)
{
  int rc = 0;
  msgvi->type = bend->hw_type;
  switch(msgvi->type) {
    case MSG_HWTYPE_EF1:
      ef_bend_hwinfo_ef1(bend, &msgvi->resources.ef1);
      break;
    case MSG_HWTYPE_NONE:
      /* Nothing to do. The slow path should just work. */
      break;
    default:
      rc = -EINVAL;
      ci_log("FATAL: don't know how to handle hardware type %d", msgvi->type);
      ef_bend_mark_dead(bend, rc, "unsupport hardware type");
      break;
  }
  return rc;
}

/* Allocate hardware resources and make them available to the client domain */
int ef_bend_setup_vnic_hw(struct ef_bend *bend)
{
  struct ef_msg msg;

  /* Allocate the event queue, VI and so on. */
  int err = ef_get_vnic(bend);
  if ( err ) {
    ci_log("Failed to allocate hardware resource for bend: error %d", err);
    ci_log("Falling back to unaccelerated mode.");
    /* Fall back to software only */
    bend->hw_type = MSG_HWTYPE_NONE;
    err = 0;
  }

  /* Set up the filter management. */
  if (err == 0) {
    err = ef_bend_filter_init(bend);
    if (err) {
              /* This is annoying but not fatal: there will be no accelerated Rx
              * path. */
      ci_log("Filter setup failed, error %d: no Rx fastpath", err);
      err = 0;
    }
  }

  ef_msg_init(&msg, EF_MSG_SETHW);

  /* Extract the low-level hardware info we will actually pass to the
  * other end, and set up the grants/ioremap permissions needed */
  ef_bend_hwinfo(bend, &msg.u.hw);


  /* Send the message */
  err = ef_msg_send_notify(bend->shared_page, bend->channel, 
                     &bend->to_domU, &msg);

  return err;
}

/* Allocate hardware resources and make them available to the client domain */
void ef_bend_shutdown_vnic_hw(struct ef_bend *bend)
{
  ci_log("Free filters...");
  ef_bend_filter_shutdown(bend);
  ef_free_vnic(bend);
}

static int allocate_iobufs(struct ef_bend *bend, int *pages)
{
  struct ef_eventq_struct *evq = bend->ep_evq;
  int rc;

  /* The code below assumes there's only one iobufset since there's
   * only one iobufs member of struct ef_bend. */
  if (bend->iobufs != NULL)
    return -EBUSY;

  /* Make sure the parameter is in range. */
  if (*pages <= 0)
    return -EINVAL;

  if (*pages > bend->quotas.max_buf_pages - bend->num_bufs) {
    int tmp = *pages;
    *pages = bend->quotas.max_buf_pages - bend->num_bufs;
    ci_log("allocation of %d pages would exceed quota %d "
        "(%d already alloc'd): truncating to %d", 
    tmp, bend->quotas.max_buf_pages,
    bend->num_bufs,
    *pages);
  }

  bend->iobufs = (ef_iobufset *)kzalloc(sizeof(ef_iobufset), GFP_KERNEL);

  if (bend->iobufs == NULL) {
    return -ENOMEM;
  }

  rc = fn_ef_iobufset_alloc(bend->iobufs, 0 /* handle */ , evq, 0 /* phys addr */,
                            PAGE_SIZE, *pages, 4 /* Align */, 0 , 0);


  if (rc < 0) {
    ci_log("ef_iobufset_alloc failed: %d", rc);
    kfree(bend->iobufs);
    bend->iobufs = NULL;
    return -ENOMEM;
  }

  bend->num_bufs += *pages;
  return 0;
}

/* Wrapper around ef_iobufset_alloc that does some logging and extracts the
 * individual pages so they can be granted to the other end. */
int ef_get_bufs( struct ef_bend *bend, int *pages, void **memaddrs,
                __u32 *bufaddr)
{
  iobufset_resource_t* iors;
  int rc;
  int i;

  /* The front end may have been disconnected and reconnected, in which case
   * it has forgotten about the buffers it had, but we are still hanging
   * onto them for it, so it will get exactly the same buf set as last time.
   * This had better be what it asks for. */
  if (bend->iobufs != NULL && bend->num_bufs == 0) {
    bend->num_bufs = iobufset_resource(bend->iobufs->bufs_mmap.rs)->n_bufs;
    if (bend->num_bufs != *pages) {
      ci_log("Warning! A VNIC has reconnected and asked for a different number"
          "of buffers from last time (%d now %d then)\nIt will not get the answer"
          "it expects", *pages, bend->num_bufs);
      *pages = bend->num_bufs;
    }
    rc = 0;
  } else {
      rc = allocate_iobufs(bend, pages);
      if (rc < 0)
        goto out;
  }

  iors = iobufset_resource(bend->iobufs->bufs_mmap.rs);

  for (i = 0; i < *pages; i++) {
    memaddrs[i] = (void *)iors->bufs[i].p.kva;
    ci_log("buffer %d: dma address " ci_dma_addr_fmt " kva %lx", i, 
           iors->bufs[i].dma_addr, iors->bufs[i].p.kva);
  }

  *bufaddr = bend->iobufs->bufs_addr;

  ci_log("buffer addr is %x", *bufaddr);
  ci_log("buffer 0 is at %p mfn %lx pfn %lx", *memaddrs, virt_to_mfn(*memaddrs), __pa(*memaddrs) >> PAGE_SHIFT);
out:
  return rc;
}


/* buffer table manipulation */
int ef_bend_buffer_table_alloc(unsigned order, 
                               efab_buffer_table_allocation *bt_handle)
{
  return fn_efab_buffer_table_alloc(order, bt_handle);
}


/* buffer table insert */
int ef_bend_buffer_table_set(struct ef_bend *bend, int mfn, int offset, 
                             efab_buffer_table_allocation *bt_handle)
{
  ci_dma_addr_t dma_addr;

  dma_addr = mfn << PAGE_SHIFT;
  
  fn_efab_buffer_table_set(bt_handle, offset, dma_addr, 
                           fn_ef_eventq_id(bend->ep_evq).index);
  /* Would be nice to not have to call commit each time, but comment
     says there are hardware restrictions on how often you can go
     without it, so do this to be safe */
  fn_efab_buffer_table_commit();

  return 0;
}
