/**************************************************************************\
*//*! \file ef_msg_iface.h Inter-VM messaging

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#ifndef EF1_MSG_INTERFACE
#define EF1_MSG_INTERFACE

#include <linux/if_ether.h>
#include <linux/spinlock.h>
#include <asm/atomic.h>
#include <asm/cache.h>
#include <asm/page.h>
#include "ci/driver/efab/open.h"
#include "ci/xen/ef_shared_fifo.h"

#define EF_MSG_MAGIC (0x85465479)


/*! Handle used to save state between lock and unlock. In Linux this
 * is an unsigned int. Move to OS abstraction header. */
typedef unsigned ef_lock_state_t;

/*! Shared memory portion of inter-VM FIFO */
struct ef_msg_queue {
  ci_uint32 read;
  ci_uint32 write;
};


/* The aflags in the following structure is used as follows:

   - each bit is set when one of the corresponding variables is
   changed by either end.

   - the end that has made the change then forwards an IRQ to the
   other

   - the IRQ handler deals with these bits either on the fast path, or
   for less common changes, by jumping onto the slow path.

   - once it has seen a change, it clears the relevant bit.

   aflags is accessed atomically using ci_bit_clear, ci_bit_test,
   ci_bit_test_and_set etc
*/

/* Split the aflags into those that can be checked on the fast path,
   and those that force us to the slow path if set */
#define MSG_AFLAGS_FASTPATH_MASK 0xffff0000
#define MSG_AFLAGS_SLOWPATH_MASK 0x0000ffff

#define MSG_AFLAGS_LINKSTATE     0x1
#define MSG_AFLAGS_LINKSTATE_B   0
#define MSG_AFLAGS_MTUCHANGE     0x2
#define MSG_AFLAGS_MTUCHANGE_B   1

/* The following used to signify to the other domain when the queue
   they want to use is full, and when it is no longer full.  Could be
   compressed to use fewer bits but done this way for simplicity and
   clarity */
#define MSG_AFLAGS_QUEUE0FULL      0x4 /* "dom0->domU queue" is full */
#define MSG_AFLAGS_QUEUE0FULL_B    2
#define MSG_AFLAGS_QUEUE0NOTFULL   0x8 /* "dom0->domU queue" is not full */
#define MSG_AFLAGS_QUEUE0NOTFULL_B 3
#define MSG_AFLAGS_QUEUEUFULL      0x10 /* "domU->dom0 queue" is full */
#define MSG_AFLAGS_QUEUEUFULL_B    4
#define MSG_AFLAGS_QUEUEUNOTFULL   0x20 /* "domU->dom0 queue" is not full */
#define MSG_AFLAGS_QUEUEUNOTFULL_B 5

/* Masks used to test if there are any messages for domU and dom0
   respectively */
#define MSG_AFLAGS_TO_DOMU_MASK (MSG_AFLAGS_LINKSTATE |         \
                                 MSG_AFLAGS_MTUCHANGE |         \
                                 MSG_AFLAGS_QUEUE0FULL |        \
                                 MSG_AFLAGS_QUEUEUNOTFULL)
#define MSG_AFLAGS_TO_DOM0_MASK (MSG_AFLAGS_QUEUE0NOTFULL |     \
                                 MSG_AFLAGS_QUEUEUFULL)


/*! The shared data structure used for inter-VM communication. */
struct ef_shared_page {
  __u32 magic;                /*!< Sanity check */
  struct ef_msg_queue queue0; /*!< Used by host/Dom0 */
  struct ef_msg_queue queue1; /*!< Used by guest/DomU */
  ci_uint32 rcv_pkts_rd;      /*!< Shared write index for the rcv_pkt fifo. */
  ci_uint32 rcv_pkts_wr;      /*!< Shared read index for the rcv_pkt fifo. */
  ci_uint32 snd_pkts_rd;      /*!< Shared write index for the snd_pkt fifo. */
  ci_uint32 snd_pkts_wr;      /*!< Shared read index for the snd_pkt fifo. */

  ci_bits  aflags;            /*!< Atomic flags, used to communicate simple state
                                changes */

  ci_uint32 link_state;       /*!< Link state - use aflags to signal changes */
  ci_uint32 mtu;              /*!< MTU - use aflags to signal changes */
};

/* The Solarflare-specific values should match the ones in driverlink_api.h,
 * hence there is a large gap before the non-Solarflare one for future use. */
/*! NIC is a EF1-based */
#define MSG_HWTYPE_EF1    (0x01)
/*! NIC is Falcon-based */
#define MSG_HWTYPE_FALCON (0x02)
/*! Not a virtualisable NIC: use slow path. */
#define MSG_HWTYPE_NONE   (0x99)

/*! The maximum number of pages used by an event queue. */
#define EF_HW_EF1_EVQ_PAGES 2

/*! EF1-specific portion of hardware description */
struct ef_hw_ef1 {
  /* VI */
  ci_uint32 txdmaq;  /*!< Page frame number for Tx DMA Q */
  ci_uint32 txbell;  /*!< Page frame number for Tx doorbell */
  ci_uint32 rxdmaq;  /*!< Page frame number for Rx DMA Q */
  ci_uint32 rxbell;  /*!< Page frame number for Rx doorbell */
  ci_uint32 txid;    /*!< Tx DMA queue number */
  ci_uint32 rxid;    /*!< Rx DMA queue number */
  /*! Event Q */
  union {
    ci_uint32 gnts[EF_HW_EF1_EVQ_PAGES]; /*!< Grants for the pages of the EVQ*/
  } evq_mem;
  ci_uint32 evq_offs;
  ci_uint32 evq_timer; /*!< Timer register physical address. Note that this is not
                        * a page number but a full address as for some odd reason we
                        * use the densely mapped timers page */
  ci_uint32 evq_order;    /*!< log2(pages in event Q) */
  ci_uint32 evq_capacity; /*!< Capacity in events */
  ci_uint32 evq_ptr;      /*!< Eventq pointer register physical address */
  ci_uint8 phys_port;     /*!< Physical port number (0 or 1) */
};

struct ef_hw_falcon {
  /* Much like EF1 except for event Qs. Also Falcon can handle
   * DMA addresses > 4G */
};

struct ef_hw_none {
  /* The Tx slowpath data structure goes here. */
};

/*! Description of the hardware that the DomU is being given. */
struct ef_msg_hw {
  unsigned type;                /*!< Hardware type */
  union {
    struct ef_hw_ef1 ef1;
    struct ef_hw_falcon falcon;
    struct ef_hw_none none;
  } resources;
};

/*! Start-of-day handshake message. Dom0 fills in its version and
 * sends, DomU checks, inserts its version and replies */
struct ef_msg_hello {
  int version;        /*!< Sender's version (set by each side in turn) */
  char mac[ETH_ALEN]; /*!< MAC address DomU should use */
};

/*! Maximum number of page requests that can fit in a message. */
#define EF_MSG_MAX_PAGE_REQ (4)

/*! Describes a page being granted. The recipient should use the grant field. */
union ef_page_req {
  ci_uint32 grant;  /*!< The grant ID */
  void *ptr;        /*!< The kernel address */
};

/*! Request for NIC buffers. DomU fils out pages and reqs (and
 *  optionally) reqid, dom0 fills out buf and sends reply */
struct ef_msg_map_buffers {
  ci_uint32 reqid;          /*!< Optional request ID */
  ci_uint32 pages;          /*!< Number of pages to map */
  union ef_page_req reqs[EF_MSG_MAX_PAGE_REQ];  /*!< Pages to map */ 
  ci_int32  mfns[EF_MSG_MAX_PAGE_REQ]; /*!< The mfns of the pages to map */
  efab_buffer_addr_t buf;   /*!< NIC buffer address of pages obtained */
};

/*! Request for NIC buffers to be added to buffer table. DomU fills
 *  out pages (and optionally) reqid, dom0 fills out reqs and buf and
 *  sends reply */
struct ef_msg_get_buffers {
  ci_uint32 reqid;          /*!< Optional request ID */
  ci_uint32 pages;          /*!< Number of pages wanted */
  union ef_page_req reqs[EF_MSG_MAX_PAGE_REQ];  /*!< Pages obtained */ 
  efab_buffer_addr_t buf;   /*!< NIC buffer address of pages obtained */
};

/*! Notification for the slow Rx path memory window. */
struct ef_msg_pktmem {
  ci_uint32 pages;  /*!< Number of pages */
  /*! page grants. "+1" here as we actually grant the first page
   *  twice, to form a partial virtual ring buffer */
  ci_uint32 grants[EF_MSG_MAX_PAGE_REQ+1];
};

/*! Notification of a change to local mac address, used to filter
  locally destined packets off the fast path */
struct ef_msg_localmac {
  ci_uint32 flags; /*!< Should this be added or removed? */
  char mac[ETH_ALEN]; /*!< The mac address to filter onto slow path */
};

/*! Values for struct ef_msg_localmac.flags */
#define EF_MSG_LOCALMAC_ADD    0x1
#define EF_MSG_LOCALMAC_REMOVE 0x2

/*! Overall message structure */
struct ef_msg {
  __u32 id;                     /*!< ID specifying type of messge */
  union {
    struct ef_msg_hello hello;  /*!< handshake */
    struct ef_msg_hw hw;        /*!< hardware description */
    struct ef_msg_get_buffers bufs; /*!< buffer request */
    struct ef_msg_map_buffers mapbufs; /*!< buffer map request */
    struct ef_msg_pktmem pktmem;  /*!< rx or tx memory window */
    struct ef_msg_localmac localmac; /*!< mac address of a local interface */
    char pad[64 - sizeof(__u32)]; /*!< make the message a fixed size */
  }  u;
};

#define MSG_HW_TO_MSG(_u) CI_CONTAINER(struct ef_msg, u.hw, _u)

/*! Inter-VM message FIFO */
typedef struct {
  struct ef_msg *fifo;
  unsigned      fifo_mask;
  ci_uint32     *fifo_rd_i;
  ci_uint32     *fifo_wr_i;
  spinlock_t    lock;
} sh_msg_fifo2;



#define EF_MSG_OFFSET_MASK PAGE_MASK

/* Modifiers */
#define EF_MSG_REPLY   (0x80000000)
#define EF_MSG_ERROR   (0x40000000)

/* Dom0 -> DomU and reply. Handshake/version check. */
#define EF_MSG_HELLO   (0x00000001)
/* Dom0 -> DomU : hardware setup (VI info.) */
#define EF_MSG_SETHW   (0x00000002)
/* DomU -> Dom0 and reply. Request for buffer table entries. */
#define EF_MSG_GETBUF  (0x00000003)
/* DomU -> Dom0. Allocation of shared pages for Rx. */
#define EF_MSG_RXMEM   (0x00000004)
/* DomU -> Dom0. Allocation of shared pages for Tx. */
#define EF_MSG_TXMEM   (0x00000005)
/* Dom0 -> DomU. Notification of a local mac to add/remove from slow
   path filter */
#define EF_MSG_LOCALMAC (0x00000006)
/* DomU -> Dom0 and reply. Request for buffer table entries for
   preallocated pages. */
#define EF_MSG_MAPBUF  (0x00000007)



/* The API needs hyperops.h for ef_notify_t. The data definitions do not. */
#include "ci/driver/virtual/ef_hyperops.h"

/*! Initialise a message and set the type
 * \param message : the message
 * \param code : the message type 
 */
static inline void ef_msg_init(struct ef_msg *msg, int code) {
  msg->id = (__u32)code;
}

/*! initialise a shared page structure
 * \param shared_page : mapped memory in which the structure resides
 * \param len : size of the message FIFO area that follows
 * \return 0 or an error code
*/
extern int ef_msg_init_page(void *shared_page, int len);

/*! initialise a message queue 
 * \param queue : the message FIFO to initialise 
 * \param indices : the read and write indices in shared memory
 * \param base : the start of the memory area for the FIFO
 * \param size : the size of the FIFO in bytes
*/
void ef_msg_init_queue(sh_msg_fifo2 *queue, struct ef_msg_queue *indices,
                       struct ef_msg *base, int size);

/*! Send a message on the specified FIFO. The message is copied to the 
 *current slot of the FIFO.
 * \param sp : pointer to shared page
 * \param q : pointer to message FIFO to use
 * \param msg : pointer to message 
 * \return 0 on success, -errno on
 */ 
extern int ef_msg_send(struct ef_shared_page *sp, sh_msg_fifo2 *q, struct ef_msg *msg);
extern int ef_msg_send_locked(struct ef_shared_page *sp, sh_msg_fifo2 *q, 
                              struct ef_msg *msg);

/*! As ef_msg_send but also posts a notification to the far end. */
extern int ef_msg_send_notify(struct ef_shared_page *sp, ef_notify_t n, sh_msg_fifo2 *q, struct ef_msg *msg);
extern int ef_msg_send_notify_locked(struct ef_shared_page *sp, ef_notify_t ev, 
                                     sh_msg_fifo2 *q, struct ef_msg *msg);

/*! Receive a message on the specified FIFO. Returns 0 on success, -errno on
 * failure. */
extern int ef_msg_recv(struct ef_shared_page *sp, sh_msg_fifo2 *q, struct ef_msg *msg);

/* Look at a received message, if any, so a decision can be made about
   whether to read it now or not.  Cookie is a bit of debug which is
   set here and checked when passed to ef_msg_recv_next() */
extern int ef_msg_peek(struct ef_shared_page *sp, sh_msg_fifo2 *queue, 
                      struct ef_msg *msg, int *cookie);
/* Move the queue onto the next element, used after finished with a
   peeked msg */
extern int ef_msg_recv_next(struct ef_shared_page *sp, sh_msg_fifo2 *queue,
                            int cookie);

/*! Start sending a message without copying. returns a pointer to a message
 * that will be filled out in place. The queue is locked until the message 
 * is sent. */
extern struct ef_msg *ef_msg_start_send(struct ef_shared_page *sp,
                                        sh_msg_fifo2 *queue, unsigned *flags);


/*! Complete the sending of a message started with ef_msg_start_send. The 
 * message is implicit since the queue was locked by _start */
extern void ef_msg_complete_send(struct ef_shared_page *sp,
                          sh_msg_fifo2 *queue,
                          unsigned *flags);

/*! As ef_msg_complete_send but does the notify. */
extern void ef_msg_complete_send_notify(struct ef_shared_page *sp, 
                                 sh_msg_fifo2 *queue,
                                 ef_lock_state_t *flags,
                                 ef_notify_t ev);

/* Give up without sending a message that was started with
   ef_msg_start_send() */
ci_inline 
void ef_msg_abort_send(struct ef_shared_page *sp, sh_msg_fifo2 *queue,
                       unsigned *flags)
{
  spin_unlock_irqrestore(&queue->lock, *flags);
}

ci_inline
void ef_msg_lock_queue(sh_msg_fifo2 *queue, unsigned *flags)
{
  spin_lock_irqsave(&queue->lock, *flags);
}

ci_inline
void ef_msg_unlock_queue(sh_msg_fifo2 *queue, unsigned *flags)
{
  spin_unlock_irqrestore(&queue->lock, *flags);
}

ci_inline
int ef_msg_check_space(sh_msg_fifo2 *queue, unsigned space)
{
  return sh_fifo2_space(queue) >= space;
}
#endif

/*! \cidoxg_end */
