#!/usr/bin/env python
'''
dump file reader commandline interface.
'''

import sys
import struct
import array

import XenCore
import ElfCore

from optparse import OptionParser

typemap = {
    'xen': XenCore.XenCoreReader,
    'elf': ElfCore.ElfCoreReader,
    }

def print_range(dump):
    '''print page frame numbers'''
    sys.stdout.write(str(dump.pages))

def print_page(dump, mfn):
    '''output raw page data'''
    content = dump.read_page(mfn)
    sys.stdout.write(content)

def main():
    '''commandline interface'''
    oparser = OptionParser()
    oparser.add_option("-p", "--mfn", dest="mfn", 
                       help="address as page number")
    oparser.add_option("-n", "--number", dest="pagenum", type="long",
                       help="address as page number")
    oparser.add_option("-a", "--addr", dest="addr", 
                       help="address")
    oparser.add_option("-r", "--print_range", dest="print_range", default=False,
                       action="store_true", help="print page number ranges")
    oparser.add_option("--arch", dest="arch", type="string", default="x86_32",
                       help="architecture ('ia64', 'x86_32', or 'x86_64')")
    oparser.add_option("-f", "--file", dest="filename", default="dump", 
                       type="string", help="dump file name")

    oparser.add_option("-t", "--type", dest="filetype", default="xen",
                       type="string", help='dump file type("xen" or "elf")')

    (options, args) = oparser.parse_args()

    # filetype, file, arch
    coretype = typemap[options.filetype]
    dump = coretype(options.filename, options.arch)

    # pagenum
    pages = 1
    if options.pagenum:
        pages = int(options.pagenum)

    # mfn
    mfn = None
    if options.mfn:
        try:
            mfn = int(options.mfn, 16)
        except:
            oparser.error("page number needs be hex")

    # address
    if options.addr:
        try:
            address = int(options.addr, 16)
            mfn = dump.arch.maddr_to_mfn(address)
        except StandardError, e:
            oparser.error("address needs be hex")
            raise e

    # output page or range
    if mfn != None:
        for i in range(pages):
            print_page(dump, mfn + i)
    elif options.print_range:
        print_range(dump)
    else:
        oparser.print_help()
        sys.exit()

if __name__ == '__main__':
    main()
