/*
 * This file is run in the DomU Kernel.
 * It grants a page with a shared ring structure on it to the Dom0.
 * The grant reference and Event Channel is passed manually. Should be done via XenStore
 * or some other out of band mechanism.
 * Compile the code using
 * make -C /lib/modules/$(shell uname -r)/build M=$(PWD) modules
 * Run it as follows
 * insmod xen-eg.ko
 * Pick up the grant ref and event channel that comes out in /var/log/messages and pass as
 * insmod parameters in the Dom0 module.
 * 
 * Run the Dom0 program as follows
 * insmod dom0.ko domid=<domid> gref=<gref>
 * <domid> is the domainid of DomU as seen in "xm list"
 * <gref> is grant refefrence when xen.ko is insmod
 */

#include <linux/module.h>
#include <linux/version.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/netdevice.h>
#include <linux/inetdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/init.h>
#include <linux/bitops.h>
#include <linux/ethtool.h>
#include <linux/in.h>
#include <linux/if_ether.h>
#include <linux/io.h>
#include <linux/moduleparam.h>
#include <net/sock.h>
#include <net/pkt_sched.h>
#include <net/arp.h>
#include <net/route.h>
#include <asm/uaccess.h>
#include <asm/page.h>
#include <xenpvdrivers/evtchn.h>
#include <xenpvdrivers/xenbus.h>
#include <xenpvdrivers/interface/io/netif.h>
#include <xenpvdrivers/interface/memory.h>
#include <xenpvdrivers/balloon.h>
#include <xenpvdrivers/asm/maddr.h>

#include <xenpvdrivers/grant_table.h>

//int page;
void *page;
struct as_request {
      unsigned int      id; /* private guest value echoed in resp */
      unsigned int      status;
      unsigned int      operation;
};

struct as_response {
      unsigned int      id; /* copied from request */
      unsigned int      status;
      unsigned int      operation; /* copied from request */
};

// The following makes the as_sring, as_back_ring, as_back_ring "types"
DEFINE_RING_TYPES(as, struct as_request, struct as_response);

struct info_t {
      struct as_front_ring    ring;
      grant_ref_t       gref;
      int               irq;
      int               port;
} info;

#define DOM0_ID 0

// Related the proc fs entries
static struct proc_dir_entry *proc_dir = NULL;
static struct proc_dir_entry *proc_file = NULL;
char proc_data[20];

#ifdef SHARED_MEM
/*
 *  Send an request via the shared ring to Dom0, following by an INT
 */

int send_request_to_dom0(void)
{
      struct as_request *ring_req;
      int notify;
      static int reqid=9;

      /* Write a request into the ring and update the req-prod pointer */
      ring_req = RING_GET_REQUEST(&(info.ring), info.ring.req_prod_pvt);
      ring_req->id = reqid;
      ring_req->operation = reqid;
      ring_req->status = reqid;
      printk("\nxen:DomU: Fill in IDX-%d, with id=%d, op=%d, st=%d",
            info.ring.req_prod_pvt, ring_req->id, ring_req->operation,
            ring_req->status);
      reqid++;
      info.ring.req_prod_pvt += 1;

      // Send a reqest to backend followed by an int if needed
      RING_PUSH_REQUESTS_AND_CHECK_NOTIFY(&(info.ring), notify);

      if (notify) {
            printk("\nxen:DomU: Sent a req to Dom0");
            notify_remote_via_irq(info.irq);
      } else {
            printk("\nxen:DomU: No notify req to Dom0");
            notify_remote_via_irq(info.irq);
      }
      printk("...\n");
      return 0;
}

ssize_t file_write (struct file *filp, const char __user *buff,
            unsigned long len, void *data)
{
      int value;

      printk("\nxen:domU: file_write %lu bytes", len);
      if (copy_from_user(&proc_data[0], buff, len))
            return -EFAULT;
      proc_data[len] = '\x0';
      //printk(" ,%s", &proc_data[0]);
      value = simple_strtol(proc_data, 0, 10);

      switch(value) {
      case 1:
            send_request_to_dom0();
            printk(" ,value = %d", value);
            break;
      default:
            printk(" ,value not recognized !");
      }
      return len;
}

int file_read (char* page, char**start, off_t off,
            int count, int *eof, void *data)
{
      sprintf(page, "%s", proc_data);
      return strlen(page);
}

/*
 *  We create a /proc/demo/file entry. When we write a "1" ino this file once
 *  the module is loaded, the file_write function() above is called and this
 *  sends a requesst on the shared ring to the Dom0. This way we test the
 *  event channel and shared ring routines.
 */
int create_procfs_entry(void)
{
      int ret = 0;

      proc_dir = proc_mkdir("demo", NULL);
      if (!proc_dir) {
            printk("\nxen:domU Could not create demo entry in procfs");
            ret = -EAGAIN;
            return ret;
      }
      proc_file = create_proc_entry("file", 0600, proc_dir);
      if (proc_file) {
            proc_file->read_proc = file_read;
            proc_file->write_proc = file_write;
#if PROC_OWNER
            proc_file->owner = THIS_MODULE;
#endif
      } else {
            printk("\nxen:domU Could not create /proc/demo/file");
            ret = -EAGAIN;
            return ret;
      }
      return ret;
}

/*
 *  Our interrupt handler for event channel that we set up
 */

static irqreturn_t as_int (int irq, void *dev_id)
{
      struct as_response *ring_resp;
      RING_IDX i, rp;
 
      printk("\nxen:DomU: as_int called");
again:
      rp = info.ring.sring->rsp_prod;
      printk("\nxen:DomU: ring pointers %d to %d", info.ring.rsp_cons, rp);
      for(i=info.ring.rsp_cons; i != rp; i++) {
            unsigned long id;
            // what did we get from Dom0
            ring_resp = RING_GET_RESPONSE(&(info.ring), i);
            printk("\nxen:DomU: Recvd in IDX-%d, with id=%d, op=%d, st=%d",
		    i, ring_resp->id, ring_resp->operation, ring_resp->status);
            id = ring_resp->id;
            switch(ring_resp->operation) {
            case 0:
                  printk("\nxen:DomU: operation:0");
                  break;
            default:
                  break;
            }
      }

      info.ring.rsp_cons = i;
      if (i != info.ring.req_prod_pvt) {
            int more_to_do;
            RING_FINAL_CHECK_FOR_RESPONSES(&info.ring, more_to_do);
            if(more_to_do)
                  goto again;
      } else
            info.ring.sring->rsp_event = i+1;
      return IRQ_HANDLED;
}
#endif

int init_module(void)
{
      int mfn;
#ifdef ENABLE_EVENT_IRQ
      int err;
#endif
      struct as_sring *sring;

/*
 *       Allocates and returns a pointer to the first byte of a memory area
 *       that is several physically contiguous pages long, and doesn't zero
 *       out the area.
 *       GFP_KERNEL - process may sleep
 */

      page = __get_free_pages(GFP_KERNEL, 1);
      if (page == 0) {
            printk("\nxen:DomU: could not get free page");
            return 0;
      }

#if ENABLE_SHARED_RING
      /* Put a shared ring structure on this page */

      sring = (struct as_sring*) page;

      SHARED_RING_INIT(sring);     

      /* info.ring is the front_ring structure */

      FRONT_RING_INIT(&(info.ring), sring, PAGE_SIZE);
#endif

 

      mfn = virt_to_mfn(page);

       /*
	*       The following grant table func is in drivers/xen/grant-table.c
	*       For shared pages, used for synchronous data, advertise a page to
	*       be shared via the hypervisor function call gnttab_grant_foreign_access.
	*       This call notifies the hypervisor that other domains are allowed to
	*       access this page.
	*    	                           
	*       gnttab_map() has been called earlier to setup gnttable_setup_table
	*       during init phase, with a call to HYPERVISOR_grant_table_op(
	*       GNTTAB_setup_table...) and
	*       "shared" pages have been malloc'ed. This "shared" page is then used
	*       below later during the actual grant of a ref by this DOM.
	*       
	*       gnttab_grant_foreign_access()
	*       => get_free_entries
	*       gnttab_free_head - points to the ref of the head
	*       gnttab_free_count- keeps number of free refs
	*
	*       Get a ref id by calling gnttab_entry(head)
	*       gnttab_list[entry/RPP][entry%RPP]
	*       => gnttab_grat_foreign_access_ref
	*       =>update_grant_entry
	*       shared[ref].frame/domid/flags are updated
	*       "shared" above is a pointer to struct grant_entry (flags/domid/frame)
	*/

      info.gref = gnttab_grant_foreign_access(DOM0_ID, mfn, 0);

      if (info.gref < 0) {

            printk("\nxen: could not grant foreign access");

            free_page((unsigned long)page);

            return 0;

      }

/*
 * The following strcpy is commented out, but was used initally to test
 * is the memory page is indeed shared with Dom0, when in Dom0, we do a
 * sprintf of the same memory location and get the same characters.
 */

       strcpy((char*)page, "aseem sethi");
/*
 * TBD: Save gref to be sent via Xenstore to dom-0. As of now both the
 * gref and the event channel port id is sent manually during insmod
 * in the dom0 module.
 */

      printk("\n gref = %d", info.gref);

      /* Setup an event channel to Dom0 */
#ifdef ENABLE_EVENT_IRQ
      err = bind_listening_port_to_irqhandler(DOM0_ID, as_int, 0,
            "xen-eg", &info);
      if (err < 0) {
            printk("\nxen:DomU failed to setup evtchn !");
            gnttab_end_foreign_access(info.gref, 0, page);
            return 0;
      }

      info.irq = err;
      info.port = irq_to_evtchn_port(info.irq);
      printk("   interupt = %d, local-port = %d", info.irq, info.port);
      printk("....\n...");
      create_procfs_entry();
#endif
      return 0;
}

void cleanup_module(void)
{
      printk("\nCleanup grant ref:");
      if (gnttab_query_foreign_access(info.gref) == 0) {
            //Remove the grant to the page
            printk("\n xen: No one has mapped this frame");

            // If 3rd param is non NULL, page has to be freed
            gnttab_end_foreign_access(info.gref, 0, page);
            // free_pages(page,1);
      } else {
            printk("\n xen: Someone has mapped this frame");
            // Guess, we still free the page, since we are rmmod-ed
            gnttab_end_foreign_access(info.gref, 0, page);
      }

      /* Cleanup proc entry */
      remove_proc_entry("file", proc_dir);
      remove_proc_entry("demo", NULL);
      printk("....\n...");
}

 

MODULE_LICENSE("GPL");

