(* 
    Store for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* XenStore node contents type *)
type ('node, 'contents) node_contents =
  | Empty
  | Value of 'contents
  | Children of 'node list
  | Hack of 'contents * 'node list

let dividor = '/'
let dividor_str = String.make 1 dividor
let root_path = dividor_str
let domain_root = root_path ^ "local" ^ dividor_str ^ "domain" ^ dividor_str
let valid_characters = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-/_@"

(* Return the base path of a path *)
let base_path path =
  if path = root_path
  then path
  else
    let start = succ (String.rindex path dividor) in
    String.sub path start ((String.length path) - start)

(* Compare two nodes *)
let compare node1 node2 =
  String.compare node1#path node2#path

(* Check if a path is a child of another path *)
let is_child child parent =
  if parent = root_path
  then true
  else
    let length = min (String.length parent) (String.length child) in
    if String.sub child 0 length <> String.sub parent 0 length
    then false
    else
      let parent_length = String.length parent
      and child_length = String.length child in
      (* XXX: This returns child = parent *)
      if parent_length = child_length
      then true
      else if parent_length < child_length then String.get child parent_length = dividor
      else false

(* Check if a path is an event path *)
let is_event path =
  path.[0] = Constants.event_char

(* Check if a path is a relative path *)
let is_relative path =
  not (is_event path) && String.sub path 0 (String.length root_path) <> root_path

(* Iterate over nodes applying function f to each node *)
let rec iter f node =
  match node#contents with
  | Empty -> ()
  | Children children -> List.iter (fun child -> iter f child) children
  | Value value -> f value
  | Hack (value, children) -> f value; List.iter (fun child -> iter f child) children

(* Return the parent path of a path *)
let parent_path path =
  let slash = String.rindex path dividor in
  if slash = 0 then root_path else String.sub path 0 slash

(* Return canonicalised path *)
let canonicalise domain path =
  if not (is_relative path) then path else domain_root ^ (string_of_int domain#id) ^ dividor_str ^ path

(* XenStore node type *)
class ['contents] node path (contents : ('contents node, 'contents) node_contents) =
object (self)
  val m_path = path
  val mutable m_contents = contents
  method add_child child =
    match self#contents with
    | Empty -> m_contents <- Children [ child ]; true
    | Value value -> m_contents <- Hack (value, [ child ]); true (* false *)
    | Children children -> m_contents <- Children (List.sort compare (child :: children)); true
    | Hack (value, children) -> m_contents <- Hack (value, List.sort compare (child :: children)); true
  method contents = m_contents
  method path = m_path
  method get_child child_path =
    match self#contents with
    | Children children | Hack (_, children) -> (
          try List.find (fun child_node -> child_node#path = child_path) children
          with Not_found -> raise (Constants.Xs_error (Constants.ENOENT, "Store.node#get_child", child_path))
        )
    | _ -> raise (Constants.Xs_error (Constants.ENOENT, "Store.node#get_child", child_path))
  method remove_child child_path =
    match self#contents with
    | Children children -> m_contents <- Children (List.filter (fun child_node -> child_node#path <> child_path) children)
    | Hack (value, children) -> m_contents <- Hack (value, List.filter (fun child_node -> child_node#path <> child_path) children)
    | _ -> raise (Constants.Xs_error (Constants.ENOENT, "Store.node#remove_child", path))
  method set_contents contents = m_contents <- contents
end

class ['contents] store =
object (self)
  val m_root : 'contents node = new node root_path (Children [])
  method private construct_node path =
    let parent_path = parent_path path in
    let parent_node = try self#get_node parent_path with _ -> self#construct_node parent_path
    and node = new node path Empty in
    if parent_node#add_child node then node else raise (Constants.Xs_error (Constants.ENOENT, "Store.store#construct_node", path))
  method private get_node path = if path = root_path then self#root else (self#get_node (parent_path path))#get_child path
  method private root = m_root
  method create_node path = ignore (self#construct_node path)
  method iter f = iter f self#root
  method node_exists path = try ignore (self#get_node path); true with _ -> false
  method read_node path = (self#get_node path)#contents
  method remove_node path = (self#get_node (parent_path path))#remove_child path
  method replace_node (node : 'contents node) =
    let node_to_replace =
      if node#path = root_path
      then self#root
      else (
        if self#node_exists node#path then self#remove_node node#path;
        self#construct_node node#path
      ) in
    node_to_replace#set_contents node#contents
  method write_node path (contents : 'contents) =
    let node = self#get_node path in
    match node#contents with
    | Empty | Value _ -> node#set_contents (Value contents)
    | Children children | Hack (_, children) -> node#set_contents (Hack (contents, children))
end
