(* 
    Connections for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

class buffer length =
object (self)
  val m_buffer = String.make length Constants.null_char
  val mutable m_position = 0
  method private position = m_position
  method buffer = String.copy m_buffer
  method clear =
    String.blit (String.make self#length Constants.null_char) 0 m_buffer 0 self#length;
    m_position <- 0
  method length = String.length m_buffer
  method remaining = self#length - self#position
  method write data =
    let length = (String.length data) in
    String.blit data 0 m_buffer m_position length;
    m_position <- m_position + length
end

class buffered_message =
object (self)
  val m_header = new buffer Message.header_size
  val mutable m_payload = new buffer 0
  method allocate_payload length = m_payload <- new buffer length
  method clear =
    self#header#clear;
    self#allocate_payload 0
  method header = m_header
  method in_header = self#header#remaining <> 0
  method in_payload = self#payload#remaining <> 0
  method message =
    if self#in_header
    then Message.null_message
    else (
      let header = Message.deserialise_header self#header#buffer in
      let payload = if self#payload#length = 0 then String.make header.Message.length Constants.null_char else self#payload#buffer in
      Message.make header.Message.message_type header.Message.transaction_id header.Message.request_id payload
    )
  method payload = m_payload
end

class connection (interface : Interface.interface) =
object (self)
  val m_input_buffer = new buffered_message
  val m_interface = interface
  method private interface = m_interface
  method private input_buffer = m_input_buffer
  method private read_buffer buffer =
    let read_buffer = String.make buffer#remaining Constants.null_char in
    let bytes = self#interface#read read_buffer 0 (String.length read_buffer) in
    if bytes < 0
    then raise (Constants.Xs_error (Constants.EIO, "Connection.connection#read_buffer", "Error reading from interface"))
    else (buffer#write (String.sub read_buffer 0 bytes); buffer#remaining = 0)
  method private write_buffer buffer offset =
    let length = String.length buffer in
    let bytes_written = self#interface#write buffer offset (length - offset) in
    if offset + bytes_written < length then self#write_buffer buffer (offset + bytes_written)
  method can_read = self#interface#can_read
  method can_write = self#interface#can_write
  method destroy = self#interface#destroy
  method read =
    let input = self#input_buffer in
    if input#in_header && self#read_buffer input#header
    then (
      let length = input#message.Message.header.Message.length in
      if length > Constants.payload_max
      then raise (Constants.Xs_error (Constants.EIO, "Connection.connection#read", "Payload too big"))
      else input#allocate_payload length
    );
    if (not input#in_header && not input#in_payload) || (input#in_payload && self#read_buffer input#payload)
    then (
      let message = input#message in
      input#clear;
      Some (message)
    )
    else None
  method write message = self#write_buffer ((Message.serialise_header message.Message.header) ^ message.Message.payload) 0
end
