/**************************************************************************\
*//*! \file ef_vnic_netdev.c 

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/
#include <linux/skbuff.h>
#include <linux/if_ether.h>
#include <linux/netdevice.h>
#include <linux/proc_fs.h>
#include <ci/driver/virtual/vnic.h>
#include "ci/xen/ef_msg_iface.h"
#include "ef_vnic_bufs.h"
#include "ef_vnic_netdev.h"


/* Forward a packet to the backend - tx slow path */
static int forward_to_bend(ef_vnic_bec *vnic_data, char *packet,
                           int packet_len, int fragment_len, 
                           int first, int last)
{
  int space;
  VNIC_VERB(ci_log("%s: %d, %d, %d, %d", __FUNCTION__, packet_len,
                   fragment_len, first, last));

  /* Check for space in the fifo */
  if(sh_fifo2_space(&vnic_data->snd_pkts) < packet_len + 2){
    ci_log("No FIFO space.  Have %d, need %d", 
           sh_fifo2_space(&vnic_data->snd_pkts), packet_len);
    return -ENOMEM;
  }

  /* Tell it how much */
  if(first){
    sh_fifo2_put(&vnic_data->snd_pkts, packet_len & 0xff);
    sh_fifo2_put(&vnic_data->snd_pkts, (packet_len >> 8) & 0xff);
  }

  space = sh_fifo2_contig_space(&vnic_data->snd_pkts);
  /* Can copy in one go due to virtual ring buffer as long as packet
     len doesn't go more than one page past the end */
  ci_assert_lt(fragment_len, PAGE_SIZE);
  ci_assert(space >= fragment_len || fragment_len - space < PAGE_SIZE);
  memcpy(sh_fifo2_pokep(&vnic_data->snd_pkts), packet, fragment_len);
  sh_fifo2_wr_adv(&vnic_data->snd_pkts, fragment_len);

  if(last){
    /* Tell the other end there is a packet waiting for it */
    ef_hyperop_remote_irq(vnic_data->net_channel);
  }

  return 0;
}




/* Request to transmit a packet */
int ef_vnic_slowpath_tx(struct sk_buff *skb, ef_vnic_svh *svh)
{
  void *data;
  int len, i;
  ef_vnic_bec *vnic = EF_VNIC_BEC_FROM_SVH(svh);
  skb_frag_t *fragment;
  struct net_device_stats *netstats = &EF_VNIC_OSD_FROM_SVH(svh)->stats;
  VNIC_VERB(ci_log("request to Tx skb %p", skb));  

  data = skb->data;
  len = skb_headlen(skb);
  i = 0;
  while(1) { /* For each fragment */
    /* Copies the data into the packet buffer */
    forward_to_bend(vnic, data, skb->len, len, data == skb->data, 
                    i >= skb_shinfo(skb)->nr_frags);
    /* Stop if we've exhausted all the fragments */
    if ( i >= skb_shinfo(skb)->nr_frags )
      break;
    /* Otherwise set up the next fragment */
    fragment = &skb_shinfo(skb)->frags[i];
    len = fragment->size;
    data = ( (void*)page_address(fragment->page) +
             fragment->page_offset );
    i++;
  }

  /* Track stats */
  netstats->tx_packets++;
  netstats->tx_bytes += skb->len;

  /* Since there's no chance of a zcopy in the slow path, just free it now. */
  dev_kfree_skb(skb);

  return 0;
}


