/**************************************************************************\
*//*! \file ef_filter.c Hardware filter insertion/removal for backend driver

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#include <linux/spinlock.h>
#include <ci/internal/pt.h>
#include <ci/driver/efab/internal.h>
#include "ef_bend.h"
#include "ef_filter.h"
#include "ef_char_bend.h"
#include "ef_iputil.h"

/* We statically allocate all the filters we are allowed at
 * startup. */
int ef_bend_filter_init(struct ef_bend *bend)
{
  ef_vi *vi = bend->ep_vi;
  int i, rc = -ENOMEM;
  spin_lock_init(&bend->filter_lock);

  ci_assert(bend->filter_res == NULL);

  /* Create the filter resources */
  bend->filter_res = kzalloc(sizeof(filter_resource_t *) *
                             bend->quotas.max_filters,
                             GFP_KERNEL );
  if (bend->filter_res == NULL) {
    ci_log("No memory for filter list.");
    goto fail_no_filters;
  }

  bend->fspecs = kzalloc(sizeof(struct ef_filter_spec) *
                            bend->quotas.max_filters,
                            GFP_KERNEL );

  if (bend->fspecs == NULL) {
    ci_log("No memory for filter spec.s");
    goto fail_no_specs;
  }

  rc = 0;

  for (i = 0; i < bend->quotas.max_filters; i++) {
    rc = ef_char_alloc_filter(vi->ep_mmap.rs->rs_handle, 
                              (filter_resource_t**) &bend->filter_res[i]);
    if (rc < 0) {
      int j;
      for (j = 0; j <= i; j++) {
        ef_char_free_filter((filter_resource_t*)bend->filter_res[j]);
      }
      ci_log("Filter allocation failed: %d", rc);
      goto fail;
    }
    bend->free_filters |= (1 << i);
  }
  /* Base mask on highest set bit in max_filters  */
  bend->filter_idx_mask = (1 << fls(bend->quotas.max_filters)) - 1;
  BEND_VERB(ci_log("filter setup: max is %x mask is %x", bend->quotas.max_filters,
            bend->filter_idx_mask));
  return 0;

fail:
 kfree(bend->fspecs);
fail_no_specs:
  kfree(bend->filter_res);
fail_no_filters:
  return rc;
}

void ef_bend_filter_shutdown(struct ef_bend *bend)
{
  int i;
  if (bend->filter_res != NULL) {
    for (i = 0; i < bend->quotas.max_filters; i++) {
      ef_char_free_filter((filter_resource_t*)bend->filter_res[i]);
    }
    kfree(bend->filter_res);
    bend->filter_res = NULL;
  }
}


/*! Suggest a filter to replace when we want to insert a new one and have
 * none free.
*/
static unsigned get_victim_filter(struct ef_bend *bend)
{
  unsigned index;
  /* We could attempt to get really clever, and may do at some point, but
  * random replacement is v. cheap and low on pathological worst cases. */
  __u32 cycles;
  ci_frc32(&cycles);
  /* Some doubt about the quality of the bottom few bits, so throw 'em
  * away */
  index = (cycles >> 4) & bend->filter_idx_mask;
  /* We don't enforce that the number of filters is a power of two, but the
   * masking gets us to within one subtraction of a valid index */
  if (index >= bend->quotas.max_filters)
    index -= bend->quotas.max_filters;
  ci_log("backend %s -> %d has no free filters. Filter %d will be evicted",
         bend->nicname,
         bend->far_end,
         index
        );
  return index;
}

/* Add a filter for the specified IP/port to the backend */
filter_resource_t *ef_bend_filter_check_add(struct ef_bend *bend, struct ef_filter_spec *filt)
{
  filter_resource_t *frs;
  struct ef_filter_spec *fs;
  unsigned next_filter;
  unsigned flags;
  int rc;

  if (bend->unaccelerated) {
    return NULL;
  }
  /* Gross layering violations-R-us */
  if (filt->proto != EF_PROTO_TCP && filt->proto != EF_PROTO_UDP) {
    return NULL;
  }

  ci_log("Will add %s filter for dst ip %08x and dst port %d,", 
         (filt->proto == EF_PROTO_TCP) ? "TCP" : "UDP",
         CI_BSWAP_BE32(filt->destip_be), CI_BSWAP_BE16(filt->destport_be));

  spin_lock_irqsave(&bend->filter_lock, flags);
  if (bend->free_filters == 0) {
    next_filter = get_victim_filter(bend);
  } else {
    next_filter = __ffs(bend->free_filters);
    clear_bit(next_filter, &bend->free_filters);
  }

  fs = &bend->fspecs[next_filter];

  /* If we are recycling a filter, the previous user may want to know */
  if (fs->eviction_notice) {
    fs->eviction_notice(fs->eviction_context);
  }
  *fs = *filt;


  frs = bend->filter_res[next_filter];

  /* Add the hardware filter. We pass in the source port and address as 0
  * (wildcard) to minimise the number of filters needed. */
  if (filt->proto == IPPROTO_TCP) {
    rc = efab_filter_resource_tcp_set(frs, 0, 0, filt->destip_be,
                                      filt->destport_be);
  } else {
    rc = efab_filter_resource_udp_set(frs, 0, 0, filt->destip_be,
                                      filt->destport_be);
  }

  if (rc != 0) {
    ci_log("Hardware filter insertion failed. Error %d", rc);
    frs = NULL;
    bend->free_filters |= (1 << next_filter);
  }
  spin_unlock_irqrestore(&bend->filter_lock, flags);
  return frs;
}

#define EF_FUNC_PTR(_name) typeof(&_name) fn_ ##_name
extern EF_FUNC_PTR(efab_filter_resource_clear);

/* Remove a filter entry for the specific device and IP/port */
void ef_bend_filter_remove(struct ef_bend *bend, filter_resource_t *frs)
{
  unsigned flags;
  int i;
  spin_lock_irqsave(&bend->filter_lock, flags);

  for (i = 0; i < bend->quotas.max_filters; i++) {
    if (bend->filter_res[i] == frs)
      break;
  }
  /* We really don't like it if people free a mismatched filter and bend. */
  ci_assert(i != bend->quotas.max_filters);
  fn_efab_filter_resource_clear(frs);
  bend->free_filters |= (1 << i);
  spin_unlock_irqrestore(&bend->filter_lock, flags);
}
