/**************************************************************************\
*//*! \file ef_hyperops_xen.h Xen version of hypervisor interface

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/

#include <xen/xenbus.h>
#include <xen/evtchn.h>
#include "ci/xen/ef_xen_util.h"

/*! Handle used for notification. In XEN this is just an int. */
typedef int ef_notify_t;

/*! Handle used to identify a VM. In XEN this is the Domain ID */
typedef int ef_vmident_t;

/*! Handle used to store information about a mapping so it can be undone */
struct ef_hyperop_mapping;

/*! Bits of information that only mean anything for a specific hypervisor
 * get stuffed in here. */
struct ef_hyperop_info {
  struct xenbus_watch domu_watch;    /*!< Notification of DomU departure */
};

/* FIXME: This doesn't belong here. */
#include "ci/xen/ef_msg_iface.h"

/*! Install an event/IRQ handler
 * \param event Identifier for the event to handle
 * \param handler Handler routine
 * \param flags OS-specific flags
 * \param name  Arbitrary text string
 * \param xdev The xenbus device context
 * \return 0 on success or an error code on failure
 */
static inline int ef_hyperop_set_evhandler( ef_notify_t event,
                                            irqreturn_t (*handler)(int , void *, struct pt_regs *),
                                            unsigned flags,
                                            const char *name,
                                            struct xenbus_device *xdev
                                          )
{
  return  bind_evtchn_to_irqhandler(event, handler,
                                    flags, name,
                                    xdev);
}

/*! Raise a remote interrupt. 
** \param irq The handle for the remote interrupt
** \return 0 on success or an error code on failure 
*/
static inline int ef_hyperop_remote_irq(ef_notify_t irq)
{
  notify_remote_via_evtchn(irq);
  return 0;
}

/*! Map a series of granted pages into a contiguous area. Maps the
** first page onto the end of the mapping to create a virtual ring buffer.
** \param dev The xenbus device context
** \param msg The message containg the grant info
** \param priv Will be filled out with private data for use in unmap
** \param msg.u.rxmem.grants The array of grants
** \param msg.u.rxmem.pages Number of pages, assumed 1 less than number of grants
** \return pointer to the start of the mapping, or NULL on failure.
*/
static inline void *ef_hyperop_map_pktmem(struct xenbus_device *dev,
                                          struct ef_msg *msg, /* FIXME: Not generic */
                                          struct ef_hyperop_mapping **priv)
{
  return ef_map_grants_contig(dev, msg->u.pktmem.grants,
                              msg->u.pktmem.pages + 1, priv);
}

/*! Unmap a set of pages mapped using ef_hyperop_map_pktmem or 
 * ef_map_grants_contig.
 * \param dev The xenbus device context
 * \param priv The private data returned by the mapping function 
 */
extern void ef_hyperop_unmap_contig(struct xenbus_device *dev, 
                                    struct ef_hyperop_mapping *priv);

/*! Make an I/O page mappable by a VM.
** \param dev The xenbus device context
** \param name A description of the page. Implementation is free to ignore this.
** \param pfn Page frame number of page to make mappable
** \param farend The VM to grant the access to.
** \param rw ReadWrite or ReadOnly grant
** \return 0 on success or an error code on failure 
*/
extern int ef_hyperop_pass_io_page(struct xenbus_device *dev,
                                   const char *name,
                                   unsigned pfn,
                                   ef_vmident_t farend, int rw);

/*! Make a memory page mappable by a VM. Note that this need not have a 
 ** different implementation from ef_hyperop_pass_io_page (but it does in XEN)
 ** \param dev The xenbus device context
 ** \param name A description of the page. Implementation is free to ignore this.
 ** \param pfn Page frame number of page to make mappable
 ** \param farend The VM to grant the access to.
 ** \return 0 on success or an error code on failure 
 */
extern int ef_hyperop_pass_mem_page(struct xenbus_device *dev, 
                             const char *name,
                             unsigned pfn,
                             ef_vmident_t farend);
