/**************************************************************************\
*//*! \file ef_cuckoo_hash.c Cuckoo hash table

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/

#ifdef __KERNEL__
#include <linux/types.h> /* needed for linux/random.h */
#include <linux/random.h>
#endif

#include <ci/tools.h>

#include "ci/xen/ef_cuckoo_hash.h"

static 
int _ef_cuckoo_hash_add(ef_cuckoo_hash_table *hashtab, ef_cuckoo_hash_key key,
                        ef_cuckoo_hash_value value, int can_rehash);

/* Sets hash function parameters.  Chooses "a" to be odd, 0 < a < 2^w
   where w is the length of the key */
static void set_hash_parameters(ef_cuckoo_hash_table *hashtab)
{
  hashtab->a0 = hashtab->a1 = 0;
#ifdef __KERNEL__
  /* Make sure random */
  get_random_bytes(&hashtab->a0, EF_CUCKOO_HASH_KEY_BYTES);
  get_random_bytes(&hashtab->a1, EF_CUCKOO_HASH_KEY_BYTES);
#else
  {
    hashtab->a0 = (ef_cuckoo_hash_key)rand() | 
      ((ef_cuckoo_hash_key)(rand() & 0x0000ffff) << 32);
    hashtab->a1 = (ef_cuckoo_hash_key)rand() |
      ((ef_cuckoo_hash_key)(rand() & 0x0000ffff) << 32);
  }
#endif

  /* Make sure odd */
  hashtab->a0 |= 1;
  hashtab->a1 |= 1;
}


static ef_cuckoo_hash_key ef_cuckoo_mac_to_key(const char *mac)
{
  ci_assert(mac != NULL);

  return (ef_cuckoo_hash_key)(mac[0])
    | (ef_cuckoo_hash_key)(mac[1]) << 8
    | (ef_cuckoo_hash_key)(mac[2]) << 16
    | (ef_cuckoo_hash_key)(mac[3]) << 24
    | (ef_cuckoo_hash_key)(mac[4]) << 32
    | (ef_cuckoo_hash_key)(mac[5]) << 40;
}


int ef_cuckoo_hash_init(ef_cuckoo_hash_table *hashtab, unsigned length_bits)
{
  char *table_mem;
  unsigned length = 1 << length_bits;

  ci_assert_lt(length_bits, sizeof(unsigned) * 8);

  /* ci_log("%s, length %d", __FUNCTION__, length);*/

#ifdef __KERNEL__
  table_mem = kzalloc(sizeof(ef_cuckoo_hash_entry) * 2 * length, GFP_KERNEL);
#else
  table_mem = malloc(sizeof(ef_cuckoo_hash_entry) * 2 * length);
#endif

  if(table_mem == NULL)
    return -ENOMEM;

  hashtab->length = length;
  hashtab->length_bits = length_bits;
  hashtab->entries = 0;

  hashtab->table0 = (ef_cuckoo_hash_entry *)table_mem;
  hashtab->table1 = (ef_cuckoo_hash_entry *)
    (table_mem + length * sizeof(ef_cuckoo_hash_entry));

  set_hash_parameters(hashtab);

  /* Zero the table */
  memset(hashtab->table0, 0, length * 2 * sizeof(ef_cuckoo_hash_entry));

  return 0;
}


void ef_cuckoo_hash_destroy(ef_cuckoo_hash_table *hashtab)
{
  /* ci_log("%s", __FUNCTION__); */

#ifdef __KERNEL__
  kfree(hashtab->table0);
#else
  free(hashtab->table0);
#endif
}


/* Want to implement (ax mod 2^w) div 2^(w-q) for odd a, 0 < a < 2^w;
 * w is the length of the key, q is the length of the hash, I think.
 * See http://www.it-c.dk/people/pagh/papers/cuckoo-jour.pdf */

static ef_cuckoo_hash
ef_cuckoo_compute_hash(ef_cuckoo_hash_table *hashtab, ef_cuckoo_hash_key key, 
                       ef_cuckoo_hash_key a)
{
  unsigned w = EF_CUCKOO_HASH_KEY_BITS;
  unsigned q = hashtab->length_bits;
  ef_cuckoo_hash hash;
  ef_cuckoo_hash_key product = (a * key);

  product = product & ((ef_cuckoo_hash_key)0xffffffff |
                       ((ef_cuckoo_hash_key)0x0000ffff << 32));

  /* div by 2^n is equivalent to "asr n" */
  /* mod by 2^n is equivalent to "and 2^(n-1)" */
  
  hash = (product & (((ef_cuckoo_hash_key)2 << w) - 1)) >> (w-q);

  ci_assert_lt(hash, hashtab->length);

  return hash;
}


static int ef_cuckoo_hash_lookup0(ef_cuckoo_hash_table *hashtab,
                                  ef_cuckoo_hash_key key,
                                  ef_cuckoo_hash_value *value)
{
  ef_cuckoo_hash hash = ef_cuckoo_compute_hash(hashtab, key, hashtab->a0);

  if((hashtab->table0[hash].state == EF_CUCKOO_HASH_STATE_OCCUPIED)
     && hashtab->table0[hash].key == key){
    *value = hashtab->table0[hash].value;
    return 1;
  }

  return 0;
}

static int ef_cuckoo_hash_lookup1(ef_cuckoo_hash_table *hashtab,
                                  ef_cuckoo_hash_key key,
                                  ef_cuckoo_hash_value *value)
{
  ef_cuckoo_hash hash = ef_cuckoo_compute_hash(hashtab, key, hashtab->a1);

  if((hashtab->table1[hash].state == EF_CUCKOO_HASH_STATE_OCCUPIED)
     && hashtab->table1[hash].key == key){
    *value = hashtab->table1[hash].value;
    return 1;
  }

  return 0;
}


int ef_cuckoo_hash_lookup(ef_cuckoo_hash_table *hashtab, const char *key,
                          ef_cuckoo_hash_value *value)
{
  ef_cuckoo_hash_key internal_key = ef_cuckoo_mac_to_key(key);

  return ef_cuckoo_hash_lookup0(hashtab, internal_key, value)
    || ef_cuckoo_hash_lookup1(hashtab, internal_key, value);
}


/* Transfer any active entries from "old_table" into hashtab */
static int ef_cuckoo_hash_transfer_entries(ef_cuckoo_hash_table *hashtab,
                                           ef_cuckoo_hash_entry *old_table,
                                           unsigned old_table_length)
{
  int i, rc;
  ef_cuckoo_hash_entry *entry;

  /* ci_log("%s", __FUNCTION__);*/

  hashtab->entries = 0;

  for(i = 0; i < old_table_length; i++){
    entry = &old_table[i];
    if(entry->state == EF_CUCKOO_HASH_STATE_OCCUPIED){
      rc = _ef_cuckoo_hash_add(hashtab, entry->key, entry->value, 0);
      if(rc != 0){
        return rc;
      }
    }
  }
  
  return 0;
}


static int ef_cuckoo_hash_rehash(ef_cuckoo_hash_table *hashtab)
{
  ef_cuckoo_hash_entry *new_table, *old_table0, *old_table1;
  ef_cuckoo_hash_key old_a0, old_a1;
  int resize = 0, original_length = hashtab->length, 
    original_entries = hashtab->entries, rc;

  ci_log("%s", __FUNCTION__);

  /* resize if one of the tables is wholly used */
  if(original_entries > original_length && hashtab->length_bits < 32)
    resize = 1;

  if(resize){
    ci_log("  Resizing table: old %d new %d", hashtab->length,
           hashtab->length * 2);

#ifdef __KERNEL__    
    new_table = kzalloc(sizeof(ef_cuckoo_hash_entry) * 4 * hashtab->length,
                        GFP_KERNEL);
#else
    new_table = malloc(sizeof(ef_cuckoo_hash_entry) * 4 * hashtab->length);
#endif
    if(new_table == 0)
      return -ENOMEM;

    hashtab->length = 2 * hashtab->length;
    hashtab->length_bits++;
  }
  else{
#ifdef __KERNEL__    
    new_table = kzalloc(sizeof(ef_cuckoo_hash_entry) * 2 * hashtab->length,
                        GFP_KERNEL);
#else
    new_table = malloc(sizeof(ef_cuckoo_hash_entry) * 2 * hashtab->length);
#endif
    if(new_table == 0)
      return -ENOMEM;
  }
    
  /* Store old tables so we can access the existing values and copy across */
  old_table0 = hashtab->table0;
  old_table1 = hashtab->table1;

  /* Store old parameters so we can revert if necessary */
  old_a0 = hashtab->a0;
  old_a1 = hashtab->a1;

  /* Point hashtab to new memory region so we can try to construct new table */
  hashtab->table0 = new_table;
  hashtab->table1 = (ef_cuckoo_hash_entry *)
    ((char *)new_table + hashtab->length * sizeof(ef_cuckoo_hash_entry));
  
  
 again:
  /* Zero the new tables */
  memset(new_table, 0, hashtab->length * 2 * sizeof(ef_cuckoo_hash_entry));

  /* Choose new parameters for the hash functions */
  set_hash_parameters(hashtab);

  /* Multiply old_table_length by 2 as the length refers to each
     table, and there are two of them.  This assumes that they are
     arranged sequentially in memory, so assert it  */
  ci_assert_equal ((char *)old_table1, ((char *)old_table0 + original_length
                                        * sizeof(ef_cuckoo_hash_entry)));
  rc = ef_cuckoo_hash_transfer_entries(hashtab, old_table0, 
                                       original_length * 2);
  if(rc < 0){
    /* Problem */
    if(rc == -ENOSPC){
      /* Wanted to rehash, but rather than recurse we can just do it here */
      ci_log("  Rehash failed, trying again");
      goto again;
    }
    else{
      /* Some other error, give up, at least restore table to how it was */
      hashtab->table0 = old_table0;
      hashtab->table1 = old_table1;
      if(resize){
        hashtab->length = hashtab->length / 2;
        hashtab->length_bits --;
      }
      hashtab->a0 = old_a0;
      hashtab->a1 = old_a1;
      hashtab->entries = original_entries;
#ifdef __KERNEL__
      kfree(new_table);
#else
      free(new_table);
#endif

      ci_log("  Rehash failed, giving up");
      
      return rc;
    }
  }

  /* Success, I think */
#ifdef __KERNEL__
  kfree(old_table0);
#else
  free(old_table0);
#endif
  
  /* We should have put all the entries from old table in the new one */
  ci_assert_equal(hashtab->entries, original_entries);

  ci_log("%s: done", __FUNCTION__);

  return 0;
}


static int 
ef_cuckoo_hash_insert_or_displace(ef_cuckoo_hash_entry *table, unsigned hash,
                                  ef_cuckoo_hash_key key, 
                                  ef_cuckoo_hash_value value,
                                  ef_cuckoo_hash_key *displaced_key, 
                                  ef_cuckoo_hash_value *displaced_value)
{
  if(table[hash].state == EF_CUCKOO_HASH_STATE_VACANT){
    table[hash].key = key;
    table[hash].value = value;
    table[hash].state = EF_CUCKOO_HASH_STATE_OCCUPIED;

    /*ci_log("%s: insert", __FUNCTION__);*/

    return 1;
  }
  else{
    *displaced_key = table[hash].key;
    *displaced_value = table[hash].value;
    table[hash].key = key;
    table[hash].value = value;

    /*ci_log("%s: displace", __FUNCTION__);*/

    return 0;
  }
}


static 
int _ef_cuckoo_hash_add(ef_cuckoo_hash_table *hashtab, ef_cuckoo_hash_key key,
                       ef_cuckoo_hash_value value, int can_rehash)
{
  int hash0, hash1, i, rc;

  /*ci_log("%s: %" CI_PRIx64 ", %u", __FUNCTION__, key, value);*/

 again:

  i = 0;
  do{
    hash0 = ef_cuckoo_compute_hash(hashtab, key, hashtab->a0);
    if(ef_cuckoo_hash_insert_or_displace(hashtab->table0, hash0, key, value, 
                                         &key, &value)){
      /* Success */
      hashtab->entries++;
      return 0;
    }
    
    /* key and value now refer to the displaced item */
    
    hash1 = ef_cuckoo_compute_hash(hashtab, key, hashtab->a1);
    if(ef_cuckoo_hash_insert_or_displace(hashtab->table1, hash1, key, value, 
                                         &key, &value)){
      /* Success */
      hashtab->entries++;
      return 0;
    }
    
    /* key and value now refer to the displaced item */
  }while( ++i < EF_CUCKOO_HASH_MAX_LOOP );

  if(can_rehash){
    if((rc = ef_cuckoo_hash_rehash(hashtab)) < 0){
      /* Give up - this will drop whichever key/value pair we have
         currently displaced on the floor */
      /*ci_log("%s: rehash FAILED! dropping key/value %" CI_PRIx64 "/%u", 
        __FUNCTION__, key, value);*/
      return rc;
    }
    goto again;
  }
  
  /* Damn, couldn't do it - bad as we've now removed some random thing
     from the table, and will just drop it on the floor.  Better would
     be to somehow revert the table to the state it was in at the
     start */
  /*ci_log("%s: failed, not rehashing, giving up", __FUNCTION__);*/

  return -ENOSPC;
}

int ef_cuckoo_hash_add(ef_cuckoo_hash_table *hashtab,
                       const char *mac, ef_cuckoo_hash_value value,
                       int can_rehash)
{
  ef_cuckoo_hash_key key = ef_cuckoo_mac_to_key(mac);
  int stored_value;

#ifndef NDEBUG
  if(ef_cuckoo_hash_lookup(hashtab, mac, &stored_value))
    return -EBUSY;
#endif

  return _ef_cuckoo_hash_add(hashtab, key, value, can_rehash);
}


int ef_cuckoo_hash_remove(ef_cuckoo_hash_table *hashtab, const char *mac)
{
  ef_cuckoo_hash hash;
  ef_cuckoo_hash_key key = ef_cuckoo_mac_to_key(mac);

  /*ci_log("%s", __FUNCTION__);*/

  hash = ef_cuckoo_compute_hash(hashtab, key, hashtab->a0);
  if((hashtab->table0[hash].state == EF_CUCKOO_HASH_STATE_OCCUPIED)
     && hashtab->table0[hash].key == key){
    hashtab->table0[hash].state = EF_CUCKOO_HASH_STATE_VACANT;
    hashtab->entries--;
    return 0;
  }
  
  hash = ef_cuckoo_compute_hash(hashtab, key, hashtab->a1);
  if((hashtab->table1[hash].state == EF_CUCKOO_HASH_STATE_OCCUPIED)
     && hashtab->table1[hash].key == key){
    hashtab->table1[hash].state = EF_CUCKOO_HASH_STATE_VACANT;
    hashtab->entries--;
    return 0;
  }
 
  return -EINVAL;
}


