/**************************************************************************\
*//*! \file ef_vnic_netdev.h Xen front end driver buffer management

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

*//*
\**************************************************************************/

#ifndef EF_VNIC_BUFS
#define EF_VNIC_BUFS

#include <ci/driver/virtual/vnic.h>

/*! Buffer descriptor structure */
typedef struct ef_vnic_pkt_desc {
  efab_buffer_addr_t pkt_buff_addr;
  void *pkt_kva;
  union {
    ef_vnic_pktbuff *pkt_os_buff;
    int next_free;
  } u;
} ef_vnic_pkt_desc;

#define EF_BUF_MAX_PAGES (384)
#define EF_BUF_PAGES_PER_BLOCK_SHIFT (4)
#define EF_BUF_PAGES_PER_BLOCK (1 << (EF_BUF_PAGES_PER_BLOCK_SHIFT))
#define EF_BUF_BUFS_PER_PAGE_SHIFT (1)
#define EF_BUF_BUFS_PER_PAGE (1 << (EF_BUF_BUFS_PER_PAGE_SHIFT))
#define EF_BUF_BUFS_PER_BLOCK_SHIFT (EF_BUF_PAGES_PER_BLOCK_SHIFT + \
                                     EF_BUF_BUFS_PER_PAGE_SHIFT)
#define EF_BUF_BUFS_PER_BLOCK (1 << EF_BUF_BUFS_PER_BLOCK_SHIFT)
#define EF_BUF_NUM_BLOCKS ((EF_BUF_MAX_PAGES+EF_BUF_PAGES_PER_BLOCK-1) / \
                           EF_BUF_PAGES_PER_BLOCK)

/*! Buffer management structure.
 * FIXME: get a real buffer manager. */
typedef struct ef_vnic_bufinfo {
  unsigned magic;
  int npages;
  int nused;
  int first_free;
  spinlock_t lock;
  struct ef_vnic_pkt_desc *(desc_blocks[EF_BUF_NUM_BLOCKS]);
  void **page_list; /* list of pages allocated for network buffers */
} ef_vnic_bufinfo;

/*! Allocate memory for the buffer manager, set up locks etc.
 *
 * \return pointer to ef_vnic_bufinfo structure that represents the
 * buffer manager
 */
extern struct ef_vnic_bufinfo *ef_vnic_init_bufs(void);

/*! Release memory for the buffer manager, etc.
 *
 * \param manager  pointer to ef_vnic_bufinfo structure that represents the
 *             buffer manager
 */
extern void ef_vnic_fini_bufs(struct ef_vnic_bufinfo *manager);

/*! Release a buffer.
 *
 * \param manager  The buffer manager which owns the buffer.
 * \param id   The buffer identifier.
 */
extern void ef_vnic_buf_put(ef_vnic_bufinfo *manager, __u16 id);


/*! Get the OSD context associated with a buffer.
 *
 * \param manager  The buffer manager which owns the buffer.
 * \param id       The buffer identifier.
 *
 * The returned value is the OSD context which is associated with this
 * buffer.
 */
extern ef_vnic_pkt_desc *ef_vnic_buf_find(ef_vnic_bufinfo *manager, __u16 id);

/*! Fill out a message request for some buffers from the back end driver
 * 
 * \param manager The buffer manager 
 * \param msg Pointer to an ef_msg to complete.
 * \return 0 on success
 */
extern int ef_vnic_buf_request(ef_vnic_bufinfo *manager,
                               struct ef_msg* msg);

/*! Fill out a message request for some buffers to be mapped by the
 *back end driver
 * 
 * \param manager The buffer manager 
 * \param msg Pointer to an ef_msg to complete.
 * \return 0 on success
 */
extern int ef_vnic_buf_map_request(struct xenbus_device *dev,
                                   ef_vnic_bufinfo *manager, struct ef_msg *msg, 
                                   int pages, int offset);


/*! Process a response to a buffer request. 
 * 
 * Deal with a received message from the back end in response to our
 * request for buffers
 * 
 * \param manager The buffer manager
 * \param msg The received message from the back end describing new
 * buffers
 * \return 0 on success
 */
extern int ef_vnic_add_bufs(struct ef_vnic_bufinfo *manager,
                            struct ef_msg *msg);


/*! Allocate a buffer from the buffer manager 
 *
 * \param manager The buffer manager data structure
 * \param id On exit, the id of the buffer allocated
 * \return Pointer to buffer descriptor.
 */
struct ef_vnic_pkt_desc *ef_vnic_buf_get(ef_vnic_bufinfo *manager, __u16 *id);

/* Debug. */
ci_inline unsigned ef_vnic_buf_nused(ef_vnic_bufinfo *manager)
{
  return manager->nused;
}

ci_inline unsigned ef_vnic_buf_ntotal(ef_vnic_bufinfo *manager)
{
  return manager->npages * EF_BUF_BUFS_PER_PAGE;
}

#endif
