/**************************************************************************\
*//*! \file ef_mcast.c Multicast handling for backend driver

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

#include "ef_bend.h"
#include "ef_filter.h"
#include "ef_mcast.h"

#include "ci/xen/ef_cuckoo_hash.h"

#define MAX_MCASTS (32)
#define MCAST_HASHTAB_BITS (3)  /* Start size of 8 per table */
#define MAX_SUBS (256)

struct ef_mcast {
  struct ef_bend **subscribers;
  int num_subs;
  spinlock_t lock;
  struct ef_filter_spec filt_spec;
  filter_resource_t *filter;
  __u8 mac[ETH_ALEN];
};

struct ef_mcast *mcasts;
spinlock_t mcast_lock;
unsigned free_mcasts;

static ef_cuckoo_hash_table mcast_hashtab;


int ef_mcast_init(void)
{
  int i;
  mcasts = kzalloc(sizeof(struct ef_mcast) * MAX_MCASTS, GFP_KERNEL);
  if (mcasts == NULL) {
    return -ENOMEM;
  }

  if(ef_cuckoo_hash_init(&mcast_hashtab, MCAST_HASHTAB_BITS) != 0){
    kfree(mcasts);
    return -ENOMEM;
  }

  spin_lock_init(&mcast_lock);

  for (i = 0; i < MAX_MCASTS; i++) {
    spin_lock_init(&mcasts[i].lock);
    mcasts[i].subscribers = kzalloc(MAX_SUBS * sizeof(struct ef_bend *),
                                    GFP_KERNEL);
    /* FIXME: should we just carry on with fewer than we wanted */
    if(mcasts[i].subscribers == NULL) {
      int j;
      for (j = 0; j < i; j++) {
        kfree(mcasts[j].subscribers);
      }
      kfree(mcasts);
      ef_cuckoo_hash_destroy(&mcast_hashtab);
      return -ENOMEM;
    }
    free_mcasts |= (1 << i);
  }

  return 0;
}

void ef_mcast_shutdown()
{
  /* This is called after ef_bend_shutdown_fwd, so the forwarding
   * path is quiet and messaging has been shut down, so we don't
   * need to take any particular care. */
  int i;
  for (i = 0; i < MAX_MCASTS; i++) {
    if (mcasts[i].subscribers != NULL) {
      kfree(mcasts[i].subscribers);
    }
  }
  kfree(mcasts);
}

static struct ef_mcast *find_mcast_by_mac(__u8 *mac)
{
  ef_lock_state_t l;
  unsigned index;
  spin_lock_irqsave(&mcast_lock, l);
  if(ef_cuckoo_hash_lookup(&mcast_hashtab, mac, &index)){
    ci_assert(memcmp(mac, mcasts[index].mac, ETH_ALEN == 0));
    spin_unlock_irqrestore(&mcast_lock, l);
    return &mcasts[index];
  }  
  else {
    spin_unlock_irqrestore(&mcast_lock, l);
    return NULL;
  }
}

static struct ef_mcast *next_free_mcast(__u8 *mac)
{
  ef_lock_state_t l;
  struct ef_mcast *mc = NULL;

  spin_lock_irqsave(&mcast_lock, l);
  if(free_mcasts != 0){
    int bit = 1, i = 0;
    do{
      if(free_mcasts & bit){
        /* Got one, nab it */
        mc = &mcasts[i];
        free_mcasts &= ~bit;
        
        /* Add to the hash table */
        ef_cuckoo_hash_add(&mcast_hashtab, mac, i, 1);
        break;
      }
      i ++;
      bit = bit << 1;
    }while(i < MAX_MCASTS);
  }
  spin_unlock_irqrestore(&mcast_lock, l);
  return mc;
}

/*! Callback that indicates the filter for this MAC has just been
 * recycled under our feet. Not much to do, but we mark it as no
 * longer having a filter so that the filter can be added at some
 * future point if slow-path traffic is seen. */
static void mcast_filter_gone(void *ctxt)
{
  struct ef_mcast *mc = (struct ef_mcast *)ctxt;
  mc->filter = NULL;
}

static void release_mcast(struct ef_mcast *mc)
{
  ef_lock_state_t l;
  int num = mc - mcasts;
  spin_lock_irqsave(&mcast_lock, l);
  free_mcasts |= (1 << num);
  spin_unlock_irqrestore(&mcast_lock, l);
}

int ef_mcast_subscribe(struct ef_bend *bend, __u8 *mac)
{
  ef_lock_state_t ls;
  struct ef_mcast *mc;

  /* Check quota */
  if (bend->quotas.max_mcasts == bend->num_mcasts)
    return -EDQUOT;
  bend->num_mcasts++;

  mc = find_mcast_by_mac(mac);
  /* If no-one is interested in that MAC yet, start a trend. */
  if (mc == NULL) {
    mc = next_free_mcast(mac);
  }
  if (mc == NULL) {
    ci_log("%s could not get a free mcast structure.", __FUNCTION__);
    return -ENOMEM;
  }
  spin_lock_irqsave(&mc->lock, ls);
  if (mc->num_subs == MAX_SUBS - 1) { /* Full: early return */
    spin_unlock_irqrestore(&mc->lock, ls);
    return -ENOBUFS;
  } else if (mc->num_subs == 0) {
    /* Previously unused: we initialise the MAC. We cannot yet insert a
    * filter as we don't know the IP and port */
    memcpy(mc->mac, mac, ETH_ALEN);
  } else if (mc->filter != NULL) {
    /* Oh dear, need to take it out as filters can only deliver to one vnic */
    ci_assert(mc->num_subs == 1);
    ef_bend_filter_remove(mc->subscribers[0], mc->filter);
    mc->filter = NULL;
  }
  ci_assert(mc->subscribers[mc->num_subs] == NULL);
  mc->subscribers[mc->num_subs] = bend;
  mc->num_subs++;
  spin_unlock_irqrestore(&mc->lock, ls);
  return 0;
}

int ef_mcast_cancel_my_subscription(struct ef_bend *bend, __u8 *mac)
{
  ef_lock_state_t ls;
  int i, j;
  int err = 0;
  struct ef_mcast *mc = find_mcast_by_mac(mac);
  if (mc == NULL) {
    /* Someone is probably confused. Log it. */
    ci_log("%s: attempt to remove subscription to non-existent MAC",
           __FUNCTION__);
    err = -ENOENT;
    goto out;
  }
  spin_lock_irqsave(&mc->lock, ls);
  for (i = 0; i < mc->num_subs; i++) {
    if (mc->subscribers[i] == bend)
      break;
  }
  if (i == mc->num_subs) { /* Again someone is confused. */
    ci_log("%s: attempt to remove non-existent subscriber", __FUNCTION__);
    err = -ENOENT;
    goto unlock_out;
  }

  bend->num_mcasts++;
  /* Shuffle everything up one to keep the array dense. */
  for (j = i; j < mc->num_subs - 1; j++) {
    mc->subscribers[j] = mc->subscribers[j + 1];
  }
  mc->subscribers[j] = NULL;
  mc->num_subs--;           /* How many left now? */
  if (mc->num_subs == 1 && mc->filt_spec.destip_be != 0) {  /* Maybe they can have the filter */
    mc->filter = ef_bend_filter_check_add(mc->subscribers[0], &mc->filt_spec);
    /* Failure to insert the filter is NOT failure to remove the mcast
    * subscriptions, so we grumble but do not propagate the error back up. */
    if (mc->filter == NULL) {
      ci_log("%s: failed to insert filter for last mcast subscriber",
             __FUNCTION__);
    }
  } else if (mc->num_subs == 0) {
    release_mcast(mc);
  }
unlock_out:
  spin_unlock_irqrestore(&mc->lock, ls);
out:
  return err;
}


/* This does two largely unrelated things:
 * - get the list of subscribed bends for a MAC
 * - associate the MAC with an IP and insert a filter if possible
 * this is ugly but convenient as we only need to do one lookup.
 */
struct ef_bend **ef_mcast_process_pkt(__u8 *mac,
                                      struct ef_filter_spec *spec,
                                      int *num_bends)
{
  struct ef_mcast *mc = find_mcast_by_mac(mac);
  /* This is perfectly OK  if no-one's subscribed*/
  if (mc == NULL) {
    *num_bends = 0;
    return NULL;
  }
  /* Check the IP and port */
  if (mc->filt_spec.destip_be == 0 && spec != NULL) {
    unsigned flags;
    mc->filt_spec = *spec;
    mc->filt_spec.eviction_notice = mcast_filter_gone;
    mc->filt_spec.eviction_context = mc;
    spin_lock_irqsave(&mc->lock,flags);
    if (mc->num_subs == 1){ /* We can have a go at accelerating this. */
      mc->filter = ef_bend_filter_check_add(mc->subscribers[0], spec);
      if (mc->filter == NULL) {
        /* Not much we can do. Just whine. */
        ci_log("%s filter insertion failed", __FUNCTION__);
      }
    }
    spin_unlock_irqrestore(&mc->lock,flags);
  }
  *num_bends = mc->num_subs;
  return mc->subscribers;
}
