'''
ELF core file image interface.
This module define ElfCoreReader, ElfCoreWriter,
and their abstruct class ElfCore
'''
import sys
import os
import re
import CoreDump
import Register
import ElfPrstatus
import libelf
import sets
import StringIO
import struct

# this pattern supports both of elf32 and elf64 core output.
COMMAND = 'LANG=C readelf -l %s' 
SECTION_PATTERN = '\s+(?P<typename>\w+)\s+(?P<offset>0x[0-9a-f]+) (?P<vaddr>0x[0-9a-f]+) (?P<paddr>0x[0-9a-f]+)\s+(?P<filesize>0x[0-9a-f]+) (?P<memsize>0x[0-9a-f]+)\s+(?P<flag>[R ][W ][E ])\s+(?P<align>(?:0x)?[0-9a-f]+)'

def roundup(x, y):
    return int(int((x+(y-1))/y)*y)

def make_note(name, desc, n_type):
    outstr = StringIO.StringIO("")

    n_namesz = len(name) + 1
    n_descsz = len(desc)

    outstr.write(struct.pack("LLL", n_namesz, n_descsz, n_type))
    outstr.write(name + '\0')
    outstr.seek(roundup(outstr.pos, 4))
    outstr.write(desc)
    outstr.seek(roundup(outstr.pos, 4))
    return outstr.getvalue()

class ElfSection:
    '''
    Elf Section header class.
    '''

    def __init__(self, typename, offset, vaddr, maddr, filesize, memsize, flag, align, arch):
        self.typename = typename
        self.offset = offset
        self.vaddr = vaddr
        self.maddr = maddr
        self.filesize = filesize
        self.memsize = memsize
        self.flag = flag
        self.align = align
        self.arch = arch

        firstmfn = self.maddr / self.arch.page_size
        
        self.pages = range(firstmfn, firstmfn + (self.memsize) / self.arch.page_size)
        self.pageoffset = dict([
            (mfn, (mfn - firstmfn) * self.arch.page_size + self.offset)
            for mfn in self.pages])

    def get_mfns(self):
        '''return all mfn in this section'''
        if self.typename == 'LOAD':
            start_mfn = self.arch.maddr_to_mfn(self.maddr)
            end_mfn = self.arch.maddr_to_mfn(self.maddr + self.memsize)
            return range(start_mfn, end_mfn)
        else:
            return []
        
    def mfn_to_offsets(self, mfn):
        '''return file offset for mfn'''
        if mfn in self.pageoffset:
            return [self.pageoffset[mfn]]
        else:
            return None


class DummySection(ElfSection):
    '''
    dummy Elf Section header class.
    '''
    def __init__(self, startmach, startvirt, nr_pages, page_offset, pages, arch):
        self.arch = arch
        self.pages = pages

        self.typename = 'LOAD'

        self.page_offset = page_offset * self.arch.page_size
        self.base_offset = 0
        self.offset = self.page_offset + self.base_offset

        self.vaddr = startvirt * self.arch.page_size
        self.maddr = startmach * self.arch.page_size
        self.filesize = nr_pages * self.arch.page_size
        self.memsize = self.filesize
        self.flag = 'RWE'
        self.align = self.arch.page_size

        self.mfn_num = {}
        for num, mfn in enumerate(pages):
            if mfn in self.mfn_num:
                self.mfn_num[mfn].append(num)
            else:
                self.mfn_num[mfn] = [num]

    def set_base_offset(self, offset):
        '''set base offset for dump data in file. offset is size of ELF header'''
        self.base_offset = offset
        self.offset = self.page_offset + self.base_offset

    def mfn_to_offsets(self, mfn):
        '''return list of file offsets for mfn.
        A mfn may be mapped into some vfn, it returns list of offsets.'''
        if not mfn in self.mfn_num:
            return None
        return [num * self.arch.page_size + self.offset for num in self.mfn_num[mfn]]
        
    def get_mfns(self):
        '''return all mfn in this section'''
        return self.pages



class ElfCore(CoreDump.CoreDump):
    '''
    Elf Core file reader interface
    '''
    def __init__(self, corefilename, arch):
        CoreDump.CoreDump.__init__(self, corefilename, arch)
        self.sections = []
        self.pages = []
        self.file = None
        self.v2m = None
        self.p2m = None
        self.note = ''
        self.nr_vcpus = 0
        self.ctxts = None

    def mfn2offsets(self, mfn):
        '''return list of file offsets for a mfn'''
        offsets = []

        for sec in self.sections:
            sec_offs = sec.mfn_to_offsets(mfn)
            if sec_offs != None:
                offsets += sec_offs
        if offsets:
            return offsets
        else:
            raise KeyError('%s doesn\'t have mfn 0x%x' % (self.corefilename, mfn))
        
    def read_page(self, mfn):
        '''return a page data'''
        offsets = self.mfn2offsets(mfn)
        offset = offsets[0]
        self.file.seek(offset)
        data = self.file.read(self.arch.page_size)
        return data
    
    def write_page(self, mfn, data):
        '''put a page into index and write data into file. header is not updated by this.'''
        offsets = self.mfn2offsets(mfn)

        for offset in offsets:
            self.file.seek(offset)
            self.file.write(data)

    def has_page(self, mfn):
        '''return a page is there or not'''
        return mfn in self.pages

    def get_pagelist(self):
        '''return a list of all available mfn'''
        return self.pages
            
    def set_context(self, ctxts):
        '''set CPU(s) num and contexts'''
        self.nr_vcpus = len(ctxts)
        self.ctxts = ctxts

    def set_registers(self, regs):
        '''overwrite user registers on context'''
        
        for i, reg in enumerate(regs):
            self.ctxts[i].set_register(reg)
            
    def set_ctrlreg(self, num, val):
        '''set all ctrlreg[num] as val '''
        # it isn't used at elf
        pass

    def set_v2m(self, v2m):
        '''set virtual to machine mapping'''
        self.v2m = v2m

        if self.p2m != None:
            self._update_sections()

    def set_p2m(self, p2m):
        '''set pfn2mfn for this dump image'''
        self.p2m = p2m
        if self.v2m != None:
            self._update_sections()

class ElfCoreReader(ElfCore):
    '''
    ELF core image file reader class.
    '''
    
    def __init__(self, corefilename, arch):
        ElfCore.__init__(self, corefilename, arch)
        self.file = file(self.corefilename, 'r')
        self.sections = self._read_sections()
        if len(self.sections) == 0:
            raise ValueError, 'section isn\'t found'
        self.pages = self._get_pages()

    def _get_pages(self):
        '''return pages from self.sections'''

        pages = []
        for sec in self.sections:
            pages += sec.get_mfns()
        return pages

    def _read_sections(self):
        '''
        read section header informations via readelf
        '''

        sections = []
        f = os.popen(COMMAND % self.corefilename, 'r')

        section_prog = re.compile(SECTION_PATTERN)

        text = f.read()
        matches = section_prog.finditer(text)
        for m in matches:
            (typename, offset, vaddr, paddr, filesize, memsize, flag, align) = m.groups()

            offset = int(offset, 16)
            vaddr = int(vaddr, 16)
            paddr = int(paddr, 16)
            filesize = int(filesize, 16)
            memsize = int(memsize, 16)
            align = int(align, 16)

            sections.append( ElfSection(typename, offset, vaddr, paddr, filesize, memsize, flag, align, self.arch) )

        return sections


class ElfCoreWriter(ElfCore):
    '''
    Xen core image file writer class.
    '''
    def __init__(self, corefilename, arch):
        ElfCore.__init__(self, corefilename, arch)
        self.file = file(corefilename, 'r+')
        self.elf = None
        self._header_size = 0
        self.note = ''                  # note string
        self.note_offset = 0            # file offset for note string
        self.phdrs = []                 # Phdr list

    def __elf_init(self):
        '''make elf struct, elf header, and phdr'''
        self.elf = libelf.begin(self.arch.elf_format, libelf.EV_CURRENT, self.file, libelf.ELF_C_WRITE, None)
        ehdr = self.elf.newehdr()

        # set elf header
        ehdr.e_type = libelf.ET_CORE
        ehdr.e_machine = self.arch.elf_machine
        ehdr.e_version = libelf.EV_CURRENT
        ehdr.e_entry = 0
        ehdr.e_flags = 0
        self.phdrs = self.elf.newphdr(1 + len(self.sections))

    def __fill_note_hdr(self):
        '''make PT_NOTE header'''

        self.note_offset = self.__get_noteoffset()
        phdr = self.phdrs[0]
        phdr.p_type = libelf.PT_NOTE
        phdr.p_flags = 0
        phdr.p_offset = self.note_offset
        phdr.p_vaddr = 0
        phdr.p_paddr = 0
        phdr.p_filesz = len(self.note)
        phdr.p_memsz = 0
        phdr.p_align = 0


    def __fill_load_hdr(self):
        '''make PT_LOAD header'''

        for i, sec in enumerate(self.sections):
            phdr = self.phdrs[1 + i]
            phdr.p_type = libelf.PT_LOAD
            phdr.p_flags = libelf.PF_R | libelf.PF_W | libelf.PF_X
            phdr.p_offset = sec.offset
            phdr.p_vaddr = sec.vaddr
            phdr.p_paddr = sec.maddr
            phdr.p_filesz = sec.filesize
            phdr.p_memsz = sec.memsize
            phdr.p_align = sec.align


    def __elf_write_notes(self):
        '''make ELF notes'''

        assert self.note_offset != 0
        
        self.file.seek(self.note_offset)
        self.file.write(self.note)

    def __elf_end(self):
        '''write elf headers.'''

        self.elf.update(libelf.ELF_C_WRITE)
        self._header_size = self.elf.end()

    def __set_note_from_ctxt(self):
        '''make note header from context register info'''

        notes = []
        for ctxt in self.ctxts:
            prstatus = ElfPrstatus.ElfPrstatus(self.arch.name)
            prstatus.set_register(ctxt.get_register())
            notes.append(make_note('CORE', str(prstatus), libelf.NT_PRSTATUS))
        self.note = ''.join(notes)

    def __get_noteoffset(self):
        '''return ELF Core data offset calculated from header info.'''
        self.__set_note_from_ctxt()
        # ELF heaer + PT_NOTE, PT_LOAD header
        return (self.elf.sizeof_ehdr +
                (1 + len(self.sections)) * self.elf.sizeof_phdr)

    def __get_dataoffset(self):
        '''return ELF Core data offset calculated from header info.'''
        return self.arch.round_pgup(self.__get_noteoffset() + len(self.note))


    def write_header(self):
        '''write header'''

        self.__elf_init()

        dataoffset = self.__get_dataoffset()

        for sec in self.sections:
            sec.set_base_offset(dataoffset)

        # make program headers
        self.__fill_note_hdr()
        self.__fill_load_hdr()

        self.__elf_write_notes()
        # write headers, set self.page_offset
        self.__elf_end()


    def fetch_pages(self, other):
        '''copy pages from other dump'''
        count = 0
        pagebuf = {}
        nullpage = '\0' * self.arch.page_size

        for mfn in self.pages:
            try: 
                pagebuf[mfn] = other.read_page(mfn)
            except:
                pagebuf[mfn] = nullpage
                sys.stderr.write("%s doesn't have mfn 0x%x, the page is filled with '\\0'\n" % (other.corefilename, mfn))

            count += 1
            if count % 1024 == 0 or count == len(self.pages):
                for mfn in pagebuf:
                    self.write_page(mfn, pagebuf[mfn]) # write mfn
                pagebuf = {}
                

class ElfCoreWriterP(ElfCoreWriter):
    def _update_sections(self):
        '''update section info (looks like dump)'''
        self.pages = self.p2m.values()    # mfn list
        self.pages.sort()
        self.sections = self.__make_sections()

    def __make_sections(self):
        '''make ELF sections (looks like machine dump)'''
        pages = range(max(self.p2m.keys())+1)
        sections = []

        mrng = [None, None]             # current pfn range
        vrng = [None, None]             # current vfn range
        mfns = None
        page_offset = 0
        
        for pfn in pages:
            if pfn == mrng[1]:
                mrng[1] += 1
                vrng[1] += 1
                if pfn in self.p2m:
                    mfns.append(self.p2m[pfn])
                else:
                    mfns.append(-1)
            else:
                if not mrng == [None, None]:
                    sections.append(
                        DummySection(startmach=mrng[0],
                                     startvirt=vrng[0],
                                     nr_pages=mrng[1] - mrng[0],
                                     page_offset=page_offset,
                                     pages=mfns,
                                     arch = self.arch,
                                     ))
                    page_offset += mrng[1] - mrng[0]
                mrng = [pfn, pfn + 1]
                vrng = [0, 1]
                if pfn in self.p2m:
                    mfns = [self.p2m[pfn]]
                else:
                    mfns = [-1]

        # last section
        sections.append(
            DummySection(startmach=mrng[0],
                         startvirt=vrng[0],
                         nr_pages=mrng[1] - mrng[0],
                         page_offset=page_offset,
                         pages=mfns,
                         arch = self.arch,
                         ))
        return sections

class ElfCoreWriterV2P(ElfCoreWriter):
    def _update_sections(self):
        '''update section info (for gdb)'''
        # make virt,pseudo-phys sections  from self.p2m, self.v2m
        self.pages = self.v2m.values()    # mfn list
        self.pages.sort()

        missed_pages = list(sets.Set(self.v2m.values()) - sets.Set(self.p2m.values())) # mfn without pfn

        missed_pages.sort()
        max_used_pfn = max(self.p2m.keys() + [-1])
        
        # mfn to pfns mapping for all mfns
        m2p = {}
        for pfn in self.p2m:
            mfn = self.p2m[pfn]
            if mfn in m2p:
                m2p[mfn].append(pfn)
            else:
                m2p[mfn] = [pfn]

        for mfn, pfn in zip(list(missed_pages),
                            range(max_used_pfn+1, max_used_pfn+1+len(missed_pages))):
            if mfn in m2p:
                m2p[mfn].append(pfn)
            else:
                m2p[mfn] = [pfn]

        vpages = self.v2m.keys()
        pagemap = [(self.v2m[vfn], m2p[self.v2m[vfn]], vfn) for vfn in vpages] # (mfn, [pfn, pfn,..], vfn)

        self.sections = self.__make_sections(pagemap)
        


    def __make_sections(self, pagemap):
        '''make sections with virtual to pseudo-physical mapping(looks like process core)'''

        # pagemap is list of (mfn, [pfn, pfn,..], vfn) .
        mach = 0
        phys = 1
        virt = 2
         
        pagemap.sort(lambda x, y: cmp(x[virt], y[virt]))
        
        sections = []                   # all sections
        page_offset = 0
        
        mrngs = None                    # current pfn range
        vrng = [None, None]             # current vfn range
        mfns = []                       # mfns for this section
        
        newrange = True
        
        for page in pagemap:
            if vrng[1] == page[virt]:
                vrng[1] += 1
                # some finished phys-ranges?
                finished = sets.Set(mrngs[1]) - sets.Set(page[phys])
                if finished:
                    if len(finished) < len(mrngs[1]):
                        # there are longer ranges
                        for endphys in finished:
                            idx = mrngs[1].index(endphys)
                            del mrngs[0][idx]
                            del mrngs[1][idx]
                    elif len(finished) == len(mrngs[1]):
                        # this is the last range(s)
                        newrange = True

                # contiguous
                mrngs[1] = [endpfn+1 for endpfn in mrngs[1]]
                mfns.append(page[mach])
            else:
                newrange = True

            if newrange:
                newrange = False
                if mrngs != None:
                    # make section
                    sections.append(DummySection(startmach=mrngs[0][0],
                                                 startvirt=vrng[0],
                                                 nr_pages=mrngs[1][0] - mrngs[0][0],
                                                 page_offset=page_offset,
                                                 pages=mfns,
                                                 arch = self.arch,
                                                 ))
                    page_offset += mrngs[1][0] - mrngs[0][0]

                mrngs = [[pfn for pfn in page[phys]],
                         [pfn+1 for pfn in page[phys]]]
                vrng = [page[virt], page[virt] + 1]
                mfns = [page[mach]]

        # last section
        sections.append(DummySection(startmach=mrngs[0][0],
                                     startvirt=vrng[0],
                                     nr_pages=mrngs[1][0] - mrngs[0][0],
                                     page_offset=page_offset,
                                     pages=mfns,
                                     arch = self.arch,
                                     ))

        return sections


class ElfCoreWriterV2M(ElfCoreWriter):
    def _update_sections(self):
        '''update section info (for gdb)'''
        # make virt,pseudo-phys sections  from self.p2m, self.v2m
        self.pages = self.v2m.values()    # mfn list
        self.pages.sort()
        
        pagemap = [(vfn, self.v2m[vfn]) for vfn in self.v2m]
        #print pagemap
        
        self.sections = self.__make_sections(pagemap)

    def __make_sections(self, pagemap):
        '''make sections with virtual to machine mapping(for xen core)'''
        # pagemap is list of (vfn, mfn) .
        virt = 0
        mach = 1
        mfn = 1
        # sort all pages info(vfn, mfn)  by 'virt' field
        pagemap.sort(lambda x, y: cmp(x[virt], y[virt]))

        sections = []                   # all sections
        page_offset = 0

        mrng = [None, None] # current range
        vrng = [None, None] # current range

        for page in pagemap:
            if mrng[1] == page[mach] and vrng[1] == page[virt]:
                # contiguous
                mrng[1] += 1
                vrng[1] += 1
                mfns.append(page[mfn])
            else:
                if mrng[0] != None:
                    # new section
                    sections.append(DummySection(
                        startmach=mrng[0], startvirt=vrng[0],
                        nr_pages=mrng[1] - mrng[0],
                        page_offset=page_offset, pages=mfns,
                        arch=self.arch))
                    page_offset += mrng[1] - mrng[0]

                mrng = [page[mach], page[mach] + 1]
                vrng = [page[virt], page[virt] + 1]
                mfns = [page[mfn]]

        # last section
        sections.append(DummySection(
            startmach=mrng[0], startvirt=vrng[0], 
            nr_pages=mrng[1] - mrng[0], page_offset=page_offset, pages=mfns,
            arch=self.arch))

        return sections

