(* 
    Main functions for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

(* Handle event *)
let handle_event xenstored =
  let port = Eventchan.pending () in
  if port <> Constants.null_file_descr
  then (
    if port = xenstored#virq_port
    then (
      let domains = xenstored#domains#cleanup in
      if List.length domains > 0
      then (
        List.iter (fun domain -> if domain#is_dying then xenstored#remove_domain domain) domains;
        xenstored#watches#fire_watches "@releaseDomain" false false
      )
    );
    if Eventchan.unmask port = - 1 then Utils.barf_perror "Failed to write to event channel"
  )
  else Utils.barf_perror ("Failed to read from event channel")

(* Handle I/O for domains *)
let handle_io xenstored =
  let handle_io_for_domain domain =
    try
      if domain#can_read then domain#read;
      if domain#has_input_message
      then (
        Trace.io domain#id "IN" (Os.get_time ()) (List.hd domain#input_messages);
        let msg_type = Message.message_type_to_string (List.hd domain#input_messages).Message.header.Message.message_type
        and msg_length = (List.hd domain#input_messages).Message.header.Message.length in
        if xenstored#options.Option.verbose then (Printf.printf "Got message %s len %d from %d\n" msg_type msg_length domain#id; flush stdout);
        Process.process xenstored domain
      );
      while domain#can_write do
        let msg_type = Message.message_type_to_string (List.hd domain#output_messages).Message.header.Message.message_type
        and msg_payload = (List.hd domain#output_messages).Message.payload in
        if xenstored#options.Option.verbose then (Printf.printf "Writing msg %s (%s) out to %d\n" msg_type msg_payload domain#id; flush stdout);
        Trace.io domain#id "OUT" (Os.get_time ()) (List.hd domain#output_messages);
        domain#write
      done
    with Constants.Xs_error (Constants.EIO, _, _) -> (
          (try if not (Domain.is_unprivileged domain) then while domain#can_write do domain#write done with _ -> ());
          xenstored#remove_domain domain;
          if Domain.is_unprivileged domain then xenstored#watches#fire_watches "@releaseDomain" false false
        )
  in
  List.iter handle_io_for_domain xenstored#domains#domains

(* Main method *)
let main =
  let options = Option.parse () in
  Option.check_options options;
  
  let store = new Store.store in
  let xenstored = new Xenstored.xenstored options store in
  
  Os.init ();
  
  let event_chan = xenstored#initialise_domains in
  
  while true do
    Os.check_connections xenstored event_chan;
    if Os.check_event_chan event_chan then handle_event xenstored;
    handle_io xenstored
  done

(* Register callback for main function *)
let _ = Callback.register "main" main
