/**************************************************************************\
*//*! \file ef_bend_netdev.c 

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA


*//*
\**************************************************************************/

#include <linux/if_ether.h>
#include <linux/proc_fs.h>
#include <linux/etherdevice.h>
#include "ef_bend.h"
#include "ci/xen/ef_msg_iface.h"
#include "ef_bend_netdev.h"
#include "ef_bend_fwd.h"


static int ef_bend_netdev_open(struct net_device *dev)
{
  static const char my_mac[] = { 0xfe, 0xff, 0xff, 0xff, 0xff, 0xff };
  BEND_VERB(struct ef_bend *bend = (struct ef_bend *)dev->priv);
  BEND_VERB(ci_log("%s: %p", __FUNCTION__, bend));
  memcpy(dev->dev_addr, my_mac, ETH_ALEN);
  return 0;
}

static int ef_bend_netdev_stop(struct net_device *dev)
{
  BEND_VERB(struct ef_bend *bend = (struct ef_bend *)dev->priv);

  BEND_VERB(ci_log("%s: %p", __FUNCTION__, bend));


  return 0;
}


static int ef_bend_netdev_xmit(struct sk_buff *skb, struct net_device *dev)
{
  struct ef_bend *bend = (struct ef_bend *)dev->priv;
  int rc;
  BEND_VERB(ci_log("%s: %p", __FUNCTION__, bend));

  bend->stats->tx_packets++;
  bend->stats->tx_bytes += skb->len;

  /* Check the length - should be ETH_ZLEN as minimum */
  if(skb->len < ETH_ZLEN) {
    int incr = ETH_ZLEN - skb->len;
    /* FIXME: We need a better strategy here. */
    if(skb->tail+incr > skb->end) {
      ci_log("%s: Decided not to cause sudden death.",
             __FUNCTION__);
      dev_kfree_skb_any(skb);
      return NETDEV_TX_OK;
    }
    ci_log("%s: Runt packet extended", __FUNCTION__);
    skb_put(skb, incr);
  }

  /* Pull a bit to look like driverlink */
  skb_pull(skb, ETH_HLEN);

  /* Just forward the packet */
  rc =  forward_to_vnic(bend, skb, 0);
  /* forward_to_vnic has copied the data */
  dev_kfree_skb_any(skb);
  return rc;
}


static struct net_device_stats *ef_bend_netdev_stats(struct net_device *dev)
{
  struct ef_bend *bend = (struct ef_bend *)dev->priv;

  return bend->stats;
}


static int ef_bend_netdev_init(struct net_device *dev)
{
  /* Assign some of dev fields sensibly for us */
  ether_setup(dev);

  /* Setup callbacks */
  dev->open = ef_bend_netdev_open;
  dev->stop = ef_bend_netdev_stop;
  dev->hard_start_xmit = ef_bend_netdev_xmit;
  dev->get_stats = ef_bend_netdev_stats;

  /* Set the owner of the network device */
  SET_MODULE_OWNER(dev);

  return 0;
}

/* Start things off, bring up the network device */
int ef_bend_netdev_probe(struct ef_bend *bend)
{
  int rc;

  bend->netdev = alloc_etherdev(0);

  if(bend->netdev == NULL){
    rc = -ENOMEM;
    goto fail;
  }

  bend->stats = kzalloc(sizeof(struct net_device_stats), GFP_KERNEL);

  if(bend->stats == NULL){
    rc = -ENOMEM;
    goto fail_free_dev;
  }

  /* Set up callback and context */
  bend->netdev->init = ef_bend_netdev_init;
  bend->netdev->priv = bend;

  /* Register device with OS */
  rc = register_netdev(bend->netdev);

  ci_log("+++ slowpath netdv is %s", bend->netdev->name);
  /* Success? */
  if(rc == 0)
    return 0;
  
  /* Some error in register netdev - free everything */
  kfree(bend->stats);
  bend->stats = NULL;

 fail_free_dev:
  kfree(bend->netdev);
  bend->netdev = NULL;

 fail:
  return rc;
}

void ef_bend_netdev_remove(struct ef_bend *bend)
{
  struct  net_device *dev = bend->netdev;
  if(dev == NULL){
    return;
  }
  bend->netdev = NULL;
  netif_tx_disable(dev);
  netif_device_detach(dev);
  unregister_netdev(dev);
  dev->priv = NULL;
  free_netdev(dev);
}

/* Get a socket buffer */
void *ef_bend_netdev_get_skb(int len, struct sk_buff **skb)
{
  /* TODO make this happen out of IRQ */
  *skb = alloc_skb(len, GFP_ATOMIC);
  
  if(*skb)
    return skb_put(*skb, len);

  return NULL;
}


/* Pass a received packet out of this virtual device into the xen
   bridge: it looks just like the bottom end of a network stack */
int ef_bend_netdev_recv(struct ef_bend *bend, struct sk_buff *skb)
{
  struct net_device *dev = bend->netdev;

  skb->dev = dev;
  skb->protocol = eth_type_trans(skb, dev);
  skb->ip_summed = CHECKSUM_UNNECESSARY; /* Hasn't seen any hardware,
                                            software is infallible */
  
  bend->stats->rx_packets ++;
  bend->stats->rx_bytes += skb->len;

  return netif_rx(skb);
}
