import SymbolTable
import PageTable
import debug_info
import XenContext
import struct
import Register
import array

class XenImage:
    '''
    Xen address space and symbols environment to analyze dump file
    '''

    def __init__(self, dump, xensyms):
        self.dump = dump
        self.arch = dump.arch
        # symbol table
        self.st = SymbolTable.SymbolTable()
        self.st.loadSymbolFromObjectFile(xensyms)

        self.pt = self.get_xen_idle_pt()

        # DWARF debug information tree
        self.dwarf = debug_info.TagTree()
        self.dwarf.buildTree(xensyms)

    def get_xen_idle_pt(self):
        '''idle domain page table'''
        
        ptaddr = self.st['idle_pg_table'] - self.arch.hypervisor_page_offset # PAGE_OFFSET 
        return PageTable.PageTable(self.dump, self.arch.maddr_to_mfn(ptaddr))

    def get_xen_pt(self):
        '''get all domain pagetables list. '''

        doms = self.get_domain_list()
        vcpus = []
        for domain in doms:
            vcpus += self.get_domain_vcpus(domain)
        pagetables = []
        for vcpu in vcpus:
            pagetables.append(self.get_vcpu_pt(vcpu, 'monitor_table'))
        return pagetables

    def get_xen_stack_bottom(self, esp):
        '''if esp is in xen virtulal space, return stack bottom.'''

        return (esp & ~self.arch.stack_size) | (self.arch.stack_size - self.dwarf.get_struct_size('cpu_info'))

    def get_xen_online_cpus(self):
        '''return list of online cpu id'''
        online_cpus = []
        cpu_online_map_t = 'L'          # XXX x86_32 specific

        cpu_online_map = self.dump.read_struct(self.pt.v2m(self.st['cpu_online_map']), cpu_online_map_t)[0]
        for i in range(self.arch.max_cpu):
            if cpu_online_map & (1<<i):
                online_cpus.append(i)
        return online_cpus

    def get_xen_crash_note_regs(self, processorid):
        '''get regs from per_cpu__crash_notes from dump.'''

        # XXX x86_32 specific implementation (come from gdb/bfd/elf32-i386.c)
        percpu_size = 4096
        crash_notes_start = self.st['per_cpu__crash_notes']
        note_hdr_size = ((12+3)/4 +     # sizeof(Elf32_Note) == 12
                         (len("CORE") + 1 + 3)/4) * 4
        reg_start = note_hdr_size + 72  # offset
        reg_end = reg_start + 68        # size

        crash_notep = crash_notes_start + percpu_size * processorid
        note = self.dump.read_addr(self.pt.v2m(crash_notep), percpu_size)
        reg = Register.Register(self.arch.name)
        reg.fromElf(note[reg_start:reg_end])

        return reg

    def get_domain_list(self):
        '''return list of all domain pointers'''

        # dom0 is start of list. domain list is sorted by domain id.
        domainp = self.dump.read_struct(self.pt.v2m(self.st['dom0']), 'L')[0] 
        domain_list = []
        domain_next_in_list_offset = self.dwarf.calc_struct_member_offset('domain', 'next_in_list')
        while domainp != 0x0:
            domain_list.append(domainp)
            domainp = self.dump.read_struct(self.pt.v2m(domainp + domain_next_in_list_offset), 'L')[0]

        return domain_list
            
    def get_domainp(self, domid):
        """return domain pointer specified by domain id.
        If there isn't domain with given domain id, raise KeyError.
        """
        
        domlst = self.get_domain_list()

        for domp in domlst:
            if self.get_domain_id(domp) == domid:
                return domp
        raise KeyError

    def get_domain_id(self, domainp):
        """get domain ID from domain pointer"""
        domain_domain_id_offset = self.dwarf.calc_struct_member_offset('domain', 'domain_id')
        domain_id = self.dump.read_struct(self.pt.v2m(domainp + domain_domain_id_offset), 'L')[0]

        return domain_id


    def get_domain_pt(self, domainp):
        '''get kernel pagetable for domain. domainp is virtual address for struct domain'''
        
        vcpu_offset = self.dwarf.calc_struct_member_offset('domain', 'vcpu') # vcpu[0]
        vcpu = self.dump.read_struct(self.pt.v2m(domainp + vcpu_offset), 'L')[0]
        return self.get_vcpu_pt(vcpu, 'guest_table')

    def get_vcpu_pt(self, vcpu, table):
        '''get VCPU pagetable.
        vcpu is vcpu_info pointer,
        table is "guest_table" or "monitor_table"'''

        table_offset = self.dwarf.calc_struct_member_offset('vcpu', 'arch')
        table_offset += self.dwarf.calc_struct_member_offset('arch_vcpu', table)

        # domain->vcpu[0]->arch.guest_table
        domain_pt_mfn = self.dump.read_struct(self.pt.v2m(vcpu + table_offset), 'L')[0]

        # domain pagetable
        return PageTable.PageTable(self.dump, domain_pt_mfn)


    def get_domain_context(self, domainp):
        '''get domain context informations'''
        vcpus = self.get_domain_vcpus(domainp)

        # vcpu->arch.vcpu_guest_context
        ctxt_offset = self.dwarf.calc_struct_member_offset('vcpu', 'arch')
        ctxt_offset += self.dwarf.calc_struct_member_offset('arch_vcpu', 'guest_context')
        ctxt_size = self.dwarf.get_struct_size('vcpu_guest_context')

        # vcpu->processor 
        processor_offset = self.dwarf.calc_struct_member_offset('vcpu', 'processor')

        # vcpu->runstate.state
        state_offset = self.dwarf.calc_struct_member_offset('vcpu', 'runstate')
        state_offset += self.dwarf.calc_struct_member_offset('vcpu_runstate_info', 'state')

        struct_cpu_info_size = self.dwarf.get_struct_size('cpu_info') 
        struct_cpu_user_regs_size = self.dwarf.get_struct_size('cpu_user_regs') 

        RUNSTATE_running = 0

        ctxts = []
        for vcpu in vcpus:
            processor = self.dump.read_struct(self.pt.v2m(vcpu + processor_offset), 'L')[0]
            state = self.dump.read_struct(self.pt.v2m(vcpu + state_offset), 'L')[0]
            context = XenContext.XenContext(self.arch.name, self.dwarf, context=self.pt.load_data(vcpu + ctxt_offset, ctxt_size))
            if state == RUNSTATE_running:              
                reg = self.get_xen_crash_note_regs(processor)
                # from xen/include/asm-x86/current.h
                cpu_infop = (reg['esp'] & ~(self.arch.stack_size-1)) | (self.arch.stack_size - struct_cpu_info_size)
                guest_reg = Register.Register(self.arch.name)
                regtxt = self.dump.read_addr(self.pt.v2m(cpu_infop), struct_cpu_user_regs_size)
                guest_reg.fromXen(regtxt)
                context.set_register(guest_reg)
            ctxts.append(context)

        return ctxts

    def get_domain_vcpus(self, domainp):
        '''return list of all vcpu pointers of a domain'''

        vcpu_offset = self.dwarf.calc_struct_member_offset('domain', 'vcpu')
        ptrsize = struct.calcsize(self.arch.ptr_format)
        domain_vcpus = [
            self.dump.read_struct(self.pt.v2m(domainp + vcpu_offset + ptrsize * i), self.arch.ptr_format)[0]
            for i in range(self.arch.max_vcpu)]
        domain_vcpus = [vcpu for vcpu in domain_vcpus if vcpu != 0]
        return domain_vcpus
        

    def get_domain_summary_string(self, domainp):
        '''print domain summary for passwd domain'''
        domain_id = self.get_domain_id(domainp)
        nr_vcpu = len(self.get_domain_vcpus(domainp))
        tot_pages_offset = self.dwarf.calc_struct_member_offset('domain', 'tot_pages')
        tot_pages = self.dump.read_struct(self.pt.v2m(domainp + tot_pages_offset), 'I')[0]
        mem_size = tot_pages * self.arch.page_size

        return "domain id=%d, %d vcpu(s), size %d MB" % (domain_id, nr_vcpu, mem_size / 1024)

    def get_max_pfn(self, domainp):
        shared_info_offset = self.dwarf.calc_struct_member_offset('domain', 'shared_info')
        shared_infop = self.dump.read_struct(self.pt.v2m(domainp + shared_info_offset), 'L')[0]
        
        max_pfn_offset = self.dwarf.calc_struct_member_offset('shared_info', 'arch')
        max_pfn_offset += self.dwarf.calc_struct_member_offset('arch_shared_info', 'max_pfn')
        
        max_pfn = self.dump.read_struct(self.pt.v2m(shared_infop + max_pfn_offset), 'L')[0]
        return max_pfn

    def get_p2m(self, domainp):
        # XXX x86 depend
        max_pfn = self.get_max_pfn(domainp)

        shared_info_offset = self.dwarf.calc_struct_member_offset('domain', 'shared_info')
        shared_infop = self.dump.read_struct(self.pt.v2m(domainp + shared_info_offset), 'L')[0]

        p2m_frame_list_list_offset = self.dwarf.calc_struct_member_offset('shared_info', 'arch')
        p2m_frame_list_list_offset += self.dwarf.calc_struct_member_offset('arch_shared_info', 'pfn_to_mfn_frame_list_list')
        
        p2m_frame_list_list_mfn = self.dump.read_struct(self.pt.v2m(shared_infop + p2m_frame_list_list_offset), 'L')[0]
        p2m_frame_list_list = array.array('L')
        p2m_frame_list_list.fromstring(self.dump.read_page(p2m_frame_list_list_mfn))

        NR_MFN_IN_PAGE = 1024           

        pfn2mfn = {}
        for page_pfn in range(0, max_pfn, NR_MFN_IN_PAGE):
            p2m_frame_list_mfn = p2m_frame_list_list[page_pfn / (NR_MFN_IN_PAGE * NR_MFN_IN_PAGE)]
            if p2m_frame_list_mfn == 0:
                continue

            p2m_frame_list = array.array('L')
            p2m_frame_list.fromstring(self.dump.read_page(p2m_frame_list_mfn))
            p2m_frame_mfn = p2m_frame_list[(page_pfn / NR_MFN_IN_PAGE) % NR_MFN_IN_PAGE]
            if p2m_frame_mfn == 0:
                continue

            p2m = array.array('L')
            p2m.fromstring(self.dump.read_page(p2m_frame_mfn))

            for page_offset, mfn in enumerate(p2m):
                if not mfn & 0x80000000L:
                    pfn2mfn[page_pfn + page_offset] = mfn
        return pfn2mfn
