#!/usr/bin/env python
'''
make dom0 dump file from xen whole machine dump file
'''

import os.path
from optparse import OptionParser
import XenCore
import ElfCore
import sets
from itertools import izip
from XenImage import XenImage

typemap = {
    ('xen', 'domain'): XenCore.XenCoreWriter,
    ('xen', 'xen'): XenCore.XenCoreWriter,
    ('elf', 'domain'): ElfCore.ElfCoreWriterV2P,
    ('elf', 'xen'): ElfCore.ElfCoreWriterV2M,
    }

def list_domain(dump, xensyms):
    xenimg = XenImage(dump, xensyms)
    domlist = xenimg.get_domain_list()
    for dom in domlist:
        print xenimg.get_domain_summary_string(dom)

def domextract(outdump, dump, xensyms, domid):
    '''analyse dump with xen-syms, arch
    to make dom0 dump file outdump.
    '''
    xenimg = XenImage(dump, xensyms)

    # get dom0 info from xen
    domp = xenimg.get_domainp(domid)
    dom_pt = xenimg.get_domain_pt(domp)
    dom_context = xenimg.get_domain_context(domp)

    # make header 
    outdump.set_context(dom_context)
    if isinstance(outdump, XenCore.XenCoreWriter):
        for count, vcpup in enumerate(xenimg.get_domain_vcpus(domp)):
            outdump.set_ctrlreg(count, 3,
                                xenimg.get_vcpu_pt(vcpup, 'guest_table').mfn
                                * outdump.arch.page_size)

    p2m = xenimg.get_p2m(domp)
    outdump.set_p2m(p2m)
    v2m = dom_pt.get_virt_to_machine()
    outdump.set_v2m(v2m)
    outdump.write_header()

    # copy pages
    outdump.fetch_pages(dump)


def xenextract(outdump, dump, xensyms):
    '''analyse dump with xen-syms, arch
    to make xen dump file outdump.
    '''
    xenimg = XenImage(dump, xensyms)

    # to make guest_context structure, use dom0 guest context. but user register values are xen's
    dom0p = xenimg.get_domainp(0)
    dom0context = xenimg.get_domain_context(dom0p)
    outdump.set_context(dom0context)

    xenregs = [xenimg.get_xen_crash_note_regs(pid) for pid in xenimg.get_xen_online_cpus()]
    outdump.set_registers(xenregs)

    xen_pts = xenimg.get_xen_pt()

    if isinstance(outdump, XenCore.XenCoreWriter):
        # set all CPU's CR3 to dom0 hypervisor pagetable
        for cpuid in range(len(xenregs)):
            outdump.set_ctrlreg(cpuid, 3, xen_pts[0].mfn * outdump.arch.page_size)

        # mfns is a set of pages contained in all domain's monitor tables in xen virt area
        mfns = sets.Set()
        for pt in xen_pts:
            v2m = pt.get_virt_to_machine()
            pages = [v2m[v] for v in v2m if dump.arch.is_xen_virt(v)]
            mfns |= sets.Set(pages)
        mfns &= sets.Set(dump.get_pagelist())
        p2m = dict(izip(xrange(len(mfns)), mfns))
        outdump.set_p2m(p2m)
        outdump.set_v2m({})

    if isinstance(outdump, ElfCore.ElfCoreWriter):
        # make v2m mapping from idle domain pagetable
        outdump.set_p2m({})
        v2m = xenimg.pt.get_virt_to_machine()
        outdump.set_v2m(v2m)

    # write header
    outdump.write_header()

    # copy pages
    outdump.fetch_pages(dump)


def main():
    '''command line interface function'''

    oparser = OptionParser()
    oparser.add_option('-o', '--outdump', dest='outdumpname', default='dom0mem', type='string', help='output dump file for output')
    oparser.add_option('-d', '--xendump', dest='dumpname', default='dump', type='string', help='xen dump file for input')
    oparser.add_option('--target', dest='target', default='0', type='string', help='extract target(domain id in decimal number or "x" for xen)')
    oparser.add_option('-x', '--xensyms', dest='xensyms', default='xen-syms', type='string', help='xen binary with symbols')
    oparser.add_option('-a', '--arch', dest='arch', type='string', default='x86_32', help="architecture ('x86_32' or 'x86_32pae')")
    oparser.add_option("-t", "--type", dest="filetype", default="xen", type="string", help='output file type("xen" or "elf")')
    oparser.add_option("-f", "--force", dest="force", default=False, action='store_true', help='overwrite output file')
    oparser.add_option("-l", "--list-domain", dest="list_domain", default=False, action='store_true', help='print domain list')

    (options, args) = oparser.parse_args()

    archnames = ['x86_32', 'x86_32pae']
    
    if not os.path.exists(options.dumpname):
        print options.dumpname, 'doesn\'t exists!'
        return
    if not os.path.exists(options.xensyms):
        print options.xensyms, 'doesn\'t exists!'
        return
    if options.arch not in archnames:
        print 'not supported architecture: ', options.arch
        return
            
    # input whole-machine dump
    dump = ElfCore.ElfCoreReader(options.dumpname, options.arch)

    if options.list_domain:
        list_domain(dump, options.xensyms)
        return

    if os.path.exists(options.outdumpname) and not (options.force):
        print options.outdumpname, 'already exists!'
        return
    else:
        file(options.outdumpname, 'w')      # create blank file


    if options.target == 'x':
        target = 'xen'
    else:
        target = 'domain'

    outputFormat = typemap[(options.filetype, target)]
    outdump = outputFormat(options.outdumpname, options.arch)

    if options.target == 'x':
        xenextract(outdump, dump, options.xensyms)
    else:
        domid = int(options.target)
        domextract(outdump, dump, options.xensyms, domid)
    
if __name__ == '__main__':
    try:
        import psyco                    
        psyco.full()
    except:
        pass

    main()
