'''
Xen core file image interface.
This module define XenCoreReader, XenCoreWriter,
and their abstruct class XenCore
'''


import CoreDump
import struct
import array

CORE_MAGIC = 0xF00FEBEDL
HEADER_FORMAT = "LLLLLL"

class XenCore(CoreDump.CoreDump):
    '''xen dump file abstract class'''
    def __init__(self, corefilename, arch):
        CoreDump.CoreDump.__init__(self, corefilename, arch)

        self.xch_magic = CORE_MAGIC     # magic number
        self.xch_nr_vcpus = 0           # numver of virtual cpus
        self.xch_nr_pages = 0           # number of pages
        self.xch_ctxt_offset = None     # context infos offset
        self.xch_index_offset = None    # pfn index offset
        self.xch_pages_offset = None    # page data offset

        self.ctxt = ''                  # CPU contexts(as string)
        self.pages = []               # pfn index (as list of numbers)
        self.pageoffsetmap = {}         # mfn -> file offset hash
        self.file = None                # dump file itself

    ## CoreDump interface
    def read_page(self, mfn):
        '''return a page data.'''

        self.file.seek(self.pageoffsetmap[mfn])
        content = self.file.read(self.arch.page_size)
        return content

    def has_page(self, mfn):
        '''return a page is there or not'''
        return self.pageoffsetmap.has_key(mfn)

    def get_pagelist(self):
        '''return a list of all available mfn'''
        return self.pageoffsetmap.keys()

    ## pageoffsetmap maintainer
    def _append_pageoffsetmap(self, index, mfn):
        '''append a new page into pageoffsetmap'''
        self.pageoffsetmap[mfn] = self.xch_pages_offset + index * self.arch.page_size
        
    def _build_pageoffsetmap(self):
        '''build self.pageoffsetmap from self.pages'''
        self.pageoffsetmap = {}

        if not self.pages:                   # no pages
            return

        for i, mfn in enumerate(self.pages):
            self._append_pageoffsetmap(i, mfn)

    def clear_pages(self):
        '''delete all pages'''
        self.pages = []
        self.pageoffsetmap = {}



class XenCoreReader(XenCore):
    '''
    Xen core image file reader class.
    '''
    
    def __init__(self, corefilename, arch):
        XenCore.__init__(self, corefilename, arch)
        self.file = file(corefilename, 'r')
        self._readheader()

    def _readheader(self):
        '''read dumpfile header items'''

        self.file.seek(0)
        header = self.file.read(struct.calcsize(HEADER_FORMAT))
        (self.xch_magic,
         self.xch_nr_vcpus,
         self.xch_nr_pages,
         self.xch_ctxt_offset,
         self.xch_index_offset,
         self.xch_pages_offset) = struct.unpack(HEADER_FORMAT, header)

        if self.xch_magic != CORE_MAGIC:
            raise ValueError, 'magic number is not valid in Xen core image format'

        self.file.seek(self.xch_ctxt_offset)
        self.ctxt = self.file.read(self.xch_index_offset-self.xch_ctxt_offset)

        pt_format = self.arch.pt_format * self.xch_nr_pages
        self.file.seek(self.xch_index_offset)
        pagetable = self.file.read(struct.calcsize(pt_format))
        self.pages = list(struct.unpack(pt_format, pagetable))
        self._build_pageoffsetmap()


class XenCoreWriter(XenCore):
    '''
    Xen core image file writer class.
    '''
    def __init__(self, corefilename, arch):
        XenCore.__init__(self, corefilename, arch)
        self.file = file(corefilename, 'r+')

    def set_context(self, nr_vcpus, ctxt):
        '''set CPU(s) num and contexts'''
        self.xch_nr_vcpus = nr_vcpus
        self.ctxt = ''.join(ctxt)       # XXX may be align problem
        self.update_offsets()

    def set_nr_pages(self, nr_pages):
        '''set number of pages'''

        self.xch_nr_pages = nr_pages
        self.update_offsets()

    def update_offsets(self):
        '''recalc header offset data.'''

        self.xch_ctxt_offset = struct.calcsize(HEADER_FORMAT)
        self.xch_index_offset = (struct.calcsize(HEADER_FORMAT) + len("".join(self.ctxt)))
        self.xch_pages_offset = self.arch.round_pgup(
            self.xch_index_offset + self.xch_nr_pages * struct.calcsize('L'))
        self._build_pageoffsetmap()

    def fetch_pages(self, other, pages):
        '''copy pages from other dump'''
        count = 0
        pagebuf = {}
        for mfn in pages:
            pagebuf[mfn] = other.read_page(mfn)
            count += 1
            if count % 1024 == 0 or count == len(pages):
                for mfn in pagebuf:
                    self.write_page(mfn, pagebuf[mfn]) # write mfn
                pagebuf = {}

    def write_page(self, mfn, data):
        '''put a page into index and write data into file. header is not updated by this.'''
        # index update
        if mfn not in self.pages:
            self.pages.append(mfn)
            self._append_pageoffsetmap(len(self.pages)-1, mfn)

        # data on disk
        self.file.seek(self.pageoffsetmap[mfn])
        self.file.write(data)

    def writeheader(self):
        '''write header'''
        self.update_offsets()

        header = struct.pack(
            HEADER_FORMAT,
            self.xch_magic, self.xch_nr_vcpus, self.xch_nr_pages,
            self.xch_ctxt_offset, self.xch_index_offset,
            self.xch_pages_offset)
        self.file.seek(0)
        self.file.write(header)

        self.file.seek(self.xch_ctxt_offset)
        self.file.write(self.ctxt)

        self.file.seek(self.xch_index_offset)
        pageindex = array.array('L')
        pageindex.fromlist(self.pages) 
        self.file.write(pageindex.tostring())


